package com.ruoyi.torrent.controller;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.*;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.ruoyi.common.utils.http.HttpUtils;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.ruoyi.common.utils.SecurityUtils;
import com.ruoyi.torrent.domain.BtTorrentAnnounce;
import com.ruoyi.torrent.domain.BtTorrentFile;
import com.ruoyi.torrent.service.IBtTorrentAnnounceService;
import com.ruoyi.torrent.service.IBtTorrentFileService;
import com.ruoyi.torrent.service.IBtTorrentService;
import com.ruoyi.torrent.util.TorrentFileUtil;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.multipart.MultipartFile;

import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.torrent.domain.BtTorrent;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * 种子主Controller
 * 
 * @author ruoyi
 * @date 2025-04-21
 */
@RestController
@RequestMapping("/system/torrent")
public class BtTorrentController extends BaseController
{
    @Autowired
    private IBtTorrentService btTorrentService;

    @Autowired
    private IBtTorrentFileService btTorrentFileService;
    @Autowired
    private IBtTorrentAnnounceService btTorrentAnnounceService;
    private static final String RECOMMEND_API = "http://127.0.0.1:5000/recommend_torrents";
    private String sendJsonPost(String url, String jsonBody) throws IOException {
        HttpURLConnection connection = (HttpURLConnection) new URL(url).openConnection();
        connection.setRequestMethod("POST");
        connection.setDoOutput(true);
        connection.setRequestProperty("Content-Type", "application/json");
        connection.setRequestProperty("Accept", "application/json");

        try (OutputStream os = connection.getOutputStream()) {
            os.write(jsonBody.getBytes(StandardCharsets.UTF_8));
        }

        try (InputStream is = connection.getInputStream()) {
            return new String(is.readAllBytes(), StandardCharsets.UTF_8);
        }
    }
    public static class TorrentRecommendationRequest {
        @JsonProperty("torrent_ids")
        private List<Long> ids;

        public List<Long> getIds() {
            return ids;
        }

        public void setIds(List<Long> ids) {
            this.ids = ids;
        }
    }

    private String torrentPath= "torrents";


    private static final String boundary = "----WebKitFormBoundary" + UUID.randomUUID().toString().replace("-", "");

    @PreAuthorize("@ss.hasPermi('system:torrent:add')")
    @Log(title = "种子主", businessType = BusinessType.INSERT)
    @PostMapping("/uploadTorrent")
    public AjaxResult uploadTorrent(@RequestParam("file") MultipartFile file) {
        try {

            // Create URL connection to Flask server
            String flaskUrl = "http://49.233.215.144:5000/parse_torrent"; // Flask server URL
            HttpURLConnection connection = (HttpURLConnection) new URL(flaskUrl).openConnection();
            connection.setRequestMethod("POST");
            connection.setDoOutput(true);
            connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + boundary);

            // Open output stream
            try (OutputStream outputStream = connection.getOutputStream()) {
                // Write the multipart form data boundary and file data
                writeFormData(outputStream, file);

                // Handle Flask response
                int responseCode = connection.getResponseCode();
                if (responseCode != 200) {
                    return AjaxResult.error("Failed to communicate with Flask server, response code: " + responseCode);
                }
                TorrentFileUtil.uploadFile(file,torrentPath+"/"+SecurityUtils.getUserId());

                // Assuming the Flask server responds with JSON, parse the response
                String responseBody = new String(connection.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
                parseTorrentData(responseBody);
                return AjaxResult.success("Torrent uploaded and processed successfully");
            }
        } catch (IOException e) {
            e.printStackTrace();
            return AjaxResult.error("Error during file upload or communication with Flask service");
        }
    }

    // Write multipart form data to the output stream
    private void writeFormData(OutputStream outputStream, MultipartFile file) throws IOException {
        // Write the start boundary
        outputStream.write(("--" + boundary + "\r\n").getBytes(StandardCharsets.UTF_8));

        // Write content-disposition header
        outputStream.write(("Content-Disposition: form-data; name=\"torrent\"; filename=\"" + file.getOriginalFilename() + "\"\r\n").getBytes(StandardCharsets.UTF_8));
        outputStream.write(("Content-Type: application/octet-stream\r\n\r\n").getBytes(StandardCharsets.UTF_8));

        // Write the file content
        file.getInputStream().transferTo(outputStream);
        outputStream.write("\r\n".getBytes(StandardCharsets.UTF_8));

        // Write the closing boundary
        outputStream.write(("--" + boundary + "--\r\n").getBytes(StandardCharsets.UTF_8));
    }


    public void parseTorrentData(String responseData) {
        try {

            // Create ObjectMapper to handle JSON parsing
            ObjectMapper objectMapper = new ObjectMapper();

            // Parse the JSON response
            JsonNode jsonNode = objectMapper.readTree(responseData);
            JsonNode btTorrentNode = jsonNode.get("btTorrent");
            JsonNode btTorrentAnnounceNode = jsonNode.get("btTorrentAnnounce");
            JsonNode btTorrentFilesNode = jsonNode.get("btTorrentFiles");


            // Convert btTorrentNode to BtTorrent object
            BtTorrent btTorrent = objectMapper.readValue(btTorrentNode.toString(), BtTorrent.class);
            btTorrent.setCreatedBy(SecurityUtils.getUsername());
            btTorrent.setUploaderId(SecurityUtils.getUserId());
            btTorrent.setFilePath(torrentPath+"/"+SecurityUtils.getUserId()+"/"+btTorrent.getName());
            btTorrentService.insertBtTorrent(btTorrent);
            Long torrentId=btTorrent.getTorrentId();


            // Convert btTorrentFilesNode to List<BtTorrentFile> using TypeReference
            List<BtTorrentFile> btTorrentFiles = objectMapper.readValue(
                    btTorrentFilesNode.toString(),
                    new TypeReference<List<BtTorrentFile>>() {});
            btTorrentFiles.forEach(btFile -> {
                btFile.setTorrentId(torrentId);
                btTorrentFileService.insertBtTorrentFile(btFile);
            }
            );
            // Convert btTorrentAnnounceNode to List<BtTorrentAnnounce>
            List<BtTorrentAnnounce> btTorrentAnnounceList = new ArrayList<>();
            if (btTorrentAnnounceNode.isArray()) {
                for (JsonNode announceNode : btTorrentAnnounceNode) {
                    // You may need to adjust this depending on your BtTorrentAnnounce class structure
                    BtTorrentAnnounce announce = objectMapper.readValue(announceNode.toString(), BtTorrentAnnounce.class);
                    btTorrentAnnounceList.add(announce);
                }
            }
            btTorrentAnnounceList.forEach(btTorrentAnnounce -> {
                        btTorrentAnnounce.setTorrentId(torrentId);
                        btTorrentAnnounceService.insertBtTorrentAnnounce(btTorrentAnnounce);
                    }
            );
        } catch (Exception e) {
            e.printStackTrace();
            // Handle the error (e.g., return null or an error message)
        }
    }

    @GetMapping("/download/{id}")
    public void downloadTorrent(@PathVariable("id") Long id, HttpServletResponse response) {
        try {
            // 1. 调用 getTorrentInfo(id) 获取文件路径
            BtTorrent btTorrent= btTorrentService.selectBtTorrentByTorrentId(id);
            String filePath=btTorrent.getFilePath();
            String fileName=btTorrent.getName();

            // 2. 使用工具方法下载文件（不删除源文件）
            TorrentFileUtil.downloadFile(response, filePath, fileName, false);

        } catch (FileNotFoundException e) {

            response.setStatus(HttpServletResponse.SC_NOT_FOUND);
            System.out.println(e.getMessage());

        } catch (Exception e) {

            response.setStatus(HttpServletResponse.SC_INTERNAL_SERVER_ERROR);
            System.out.println(e.getMessage());

        }

    }

    /**
     * 查询种子主列表
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:list')")
    @GetMapping("/list")
    public TableDataInfo list(BtTorrent btTorrent)
    {
        startPage();
        List<BtTorrent> list = btTorrentService.selectBtTorrentList(btTorrent);
        return getDataTable(list);
    }

    /**
     * 查询种子主列表
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:list')")
    @PostMapping("/recommend")
    public TableDataInfo recommendTorrents() {
        // 假设当前用户的 user_id 是通过 SecurityContext 或 session 中获取的
        String userId = String.valueOf(getUserId());  // 获取当前用户的 user_id

        if (userId == null || userId.isEmpty()) {
            TableDataInfo error = new TableDataInfo();
            error.setCode(400);
            error.setMsg("用户ID无效或未登录");
            return error;
        }

        System.out.println("当前用户ID: " + userId);  // 打印调试信息

        try {
            // 1. 构造 JSON 请求体，包含用户ID
            String jsonRequest = "{\"user_id\": \"" + userId + "\"}";

            // 2. 向 Flask 服务发 POST 请求，返回 JSON 响应
            String jsonResponse = sendJsonPost(RECOMMEND_API, jsonRequest);

            // 3. 使用 Jackson 解析 JSON 响应为 TorrentRecommendationRequest 对象
            ObjectMapper mapper = new ObjectMapper();
            TorrentRecommendationRequest recommendationRequest = mapper.readValue(jsonResponse, TorrentRecommendationRequest.class);

            // 4. 提取 ids 字段
            List<Long> idList = recommendationRequest.getIds();

            if (idList == null || idList.isEmpty()) {
                throw new RuntimeException("推荐的种子ID列表为空");
            }

            // 5. 根据种子ID查询完整的种子信息，保持推荐顺序
            List<BtTorrent> resultList = btTorrentService.selectBtTorrentsByIdsOrdered(idList);

            // 6. 封装查询结果并返回
            TableDataInfo rsp = new TableDataInfo();
            rsp.setCode(200);
            rsp.setMsg("推荐成功");
            rsp.setRows(resultList);
            rsp.setTotal(resultList.size());
            return rsp;

        } catch (Exception e) {
            e.printStackTrace();  // 打印异常信息

            // 如果出错，返回错误信息
            TableDataInfo error = new TableDataInfo();
            error.setCode(500);
            error.setMsg("推荐失败：" + e.getMessage());
            return error;
        }
    }


    /**
     * 导出种子主列表
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:export')")
    @Log(title = "种子主", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, BtTorrent btTorrent)
    {
        List<BtTorrent> list = btTorrentService.selectBtTorrentList(btTorrent);
        ExcelUtil<BtTorrent> util = new ExcelUtil<BtTorrent>(BtTorrent.class);
        util.exportExcel(response, list, "种子主数据");
    }

    /**
     * 获取种子主详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:query')")
    @GetMapping(value = "/{torrentId}")
    public AjaxResult getInfo(@PathVariable("torrentId") Long torrentId)
    {
        return success(btTorrentService.selectBtTorrentByTorrentId(torrentId));
    }

    /**
     * 新增种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:add')")
    @Log(title = "种子主", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody BtTorrent btTorrent)
    {
        return toAjax(btTorrentService.insertBtTorrent(btTorrent));
    }

    /**
     * 修改种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:edit')")
    @Log(title = "种子主", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody BtTorrent btTorrent)
    {
        return toAjax(btTorrentService.updateBtTorrent(btTorrent));
    }

    /**
     * 删除种子主
     */
    @PreAuthorize("@ss.hasPermi('system:torrent:remove')")
    @Log(title = "种子主", businessType = BusinessType.DELETE)
	@DeleteMapping("/{torrentIds}")
    public AjaxResult remove(@PathVariable Long[] torrentIds)
    {
        for (Long torrentId : torrentIds){
            String filePath = btTorrentService.selectBtTorrentByTorrentId(torrentId).getFilePath();
            TorrentFileUtil.deleteFileOrFolder(filePath);
        }
        return toAjax(btTorrentService.deleteBtTorrentByTorrentIds(torrentIds));
    }
}
