package com.ruoyi.torrent.util;

import com.aliyun.oss.*;
import com.aliyun.oss.common.auth.CredentialsProviderFactory;
import com.aliyun.oss.common.auth.EnvironmentVariableCredentialsProvider;
import com.aliyun.oss.common.comm.SignVersion;
import com.aliyun.oss.internal.Mimetypes;
import com.aliyun.oss.model.*;
import com.aliyuncs.exceptions.ClientException;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

public class OssMultipartUploader {

    private final OSS ossClient;
    private final String bucketName;

    public OssMultipartUploader(String endpoint, String region, String bucketName) throws ClientException {
        EnvironmentVariableCredentialsProvider credentialsProvider =
                CredentialsProviderFactory.newEnvironmentVariableCredentialsProvider();
        ClientBuilderConfiguration config = new ClientBuilderConfiguration();
        config.setSignatureVersion(SignVersion.V4);

        this.ossClient = OSSClientBuilder.create()
                .endpoint(endpoint)
                .region(region)
                .credentialsProvider(credentialsProvider)
                .clientConfiguration(config)
                .build();

        this.bucketName = bucketName;
    }

    public void uploadFile(String objectName, String filePath) throws Exception {
        File file = new File(filePath);
        long partSize = 1 * 1024 * 1024L; // 1MB
        long fileLength = file.length();
        int partCount = (int) (fileLength / partSize);
        if (fileLength % partSize != 0) {
            partCount++;
        }

        // 设置元数据（如 content-type）
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentType(Mimetypes.getInstance().getMimetype(file, objectName));

        InitiateMultipartUploadRequest initiateRequest = new InitiateMultipartUploadRequest(bucketName, objectName);
        initiateRequest.setObjectMetadata(metadata);
        InitiateMultipartUploadResult initResult = ossClient.initiateMultipartUpload(initiateRequest);
        String uploadId = initResult.getUploadId();

        List<PartETag> partETags = new ArrayList<>();

        try {
            for (int i = 0; i < partCount; i++) {
                long startPos = i * partSize;
                long curPartSize = (i + 1 == partCount) ? (fileLength - startPos) : partSize;

                try (InputStream inputStream = new FileInputStream(file)) {
                    inputStream.skip(startPos);

                    UploadPartRequest uploadPartRequest = new UploadPartRequest();
                    uploadPartRequest.setBucketName(bucketName);
                    uploadPartRequest.setKey(objectName);
                    uploadPartRequest.setUploadId(uploadId);
                    uploadPartRequest.setInputStream(inputStream);
                    uploadPartRequest.setPartSize(curPartSize);
                    uploadPartRequest.setPartNumber(i + 1);

                    UploadPartResult uploadPartResult = ossClient.uploadPart(uploadPartRequest);
                    partETags.add(uploadPartResult.getPartETag());
                }
            }

            CompleteMultipartUploadRequest completeRequest =
                    new CompleteMultipartUploadRequest(bucketName, objectName, uploadId, partETags);

            CompleteMultipartUploadResult result = ossClient.completeMultipartUpload(completeRequest);
            System.out.println("上传成功，ETag: " + result.getETag());

        } catch (Exception e) {
            ossClient.abortMultipartUpload(new AbortMultipartUploadRequest(bucketName, objectName, uploadId));
            throw new RuntimeException("上传失败，已中止上传任务", e);
        }
    }

    public void shutdown() {
        if (ossClient != null) {
            ossClient.shutdown();
        }
    }
}
