package com.ruoyi.tracker.controller;

import java.util.List;
import jakarta.servlet.http.HttpServletResponse;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.DeleteMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;
import com.ruoyi.common.annotation.Log;
import com.ruoyi.common.core.controller.BaseController;
import com.ruoyi.common.core.domain.AjaxResult;
import com.ruoyi.common.enums.BusinessType;
import com.ruoyi.tracker.domain.TrackerTaskLog;
import com.ruoyi.tracker.service.ITrackerTaskLogService;
import com.ruoyi.common.utils.poi.ExcelUtil;
import com.ruoyi.common.core.page.TableDataInfo;

/**
 * 任务日志Controller
 * 
 * @author ruoyi
 * @date 2025-04-20
 */
@RestController
@RequestMapping("/tracker/log")
public class TrackerTaskLogController extends BaseController
{
    @Autowired
    private ITrackerTaskLogService trackerTaskLogService;

    /**
     * 查询任务日志列表
     */
    @PreAuthorize("@ss.hasPermi('system:log:list')")
    @GetMapping("/list")
    public TableDataInfo list(TrackerTaskLog trackerTaskLog)
    {
        startPage();
        List<TrackerTaskLog> list = trackerTaskLogService.selectTrackerTaskLogList(trackerTaskLog);
        return getDataTable(list);
    }

    /**
     * 导出任务日志列表
     */
    @PreAuthorize("@ss.hasPermi('system:log:export')")
    @Log(title = "任务日志", businessType = BusinessType.EXPORT)
    @PostMapping("/export")
    public void export(HttpServletResponse response, TrackerTaskLog trackerTaskLog)
    {
        List<TrackerTaskLog> list = trackerTaskLogService.selectTrackerTaskLogList(trackerTaskLog);
        ExcelUtil<TrackerTaskLog> util = new ExcelUtil<TrackerTaskLog>(TrackerTaskLog.class);
        util.exportExcel(response, list, "任务日志数据");
    }

    /**
     * 获取任务日志详细信息
     */
    @PreAuthorize("@ss.hasPermi('system:log:query')")
    @GetMapping(value = "/{logId}")
    public AjaxResult getInfo(@PathVariable("logId") Long logId)
    {
        return success(trackerTaskLogService.selectTrackerTaskLogByLogId(logId));
    }

    /**
     * 新增任务日志
     */
    @PreAuthorize("@ss.hasPermi('system:log:add')")
    @Log(title = "任务日志", businessType = BusinessType.INSERT)
    @PostMapping
    public AjaxResult add(@RequestBody TrackerTaskLog trackerTaskLog)
    {
        return toAjax(trackerTaskLogService.insertTrackerTaskLog(trackerTaskLog));
    }

    /**
     * 修改任务日志
     */
    @PreAuthorize("@ss.hasPermi('system:log:edit')")
    @Log(title = "任务日志", businessType = BusinessType.UPDATE)
    @PutMapping
    public AjaxResult edit(@RequestBody TrackerTaskLog trackerTaskLog)
    {
        return toAjax(trackerTaskLogService.updateTrackerTaskLog(trackerTaskLog));
    }

    /**
     * 删除任务日志
     */
    @PreAuthorize("@ss.hasPermi('system:log:remove')")
    @Log(title = "任务日志", businessType = BusinessType.DELETE)
	@DeleteMapping("/{logIds}")
    public AjaxResult remove(@PathVariable Long[] logIds)
    {
        return toAjax(trackerTaskLogService.deleteTrackerTaskLogByLogIds(logIds));
    }
}
