package com.example.g8backend.service;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.example.g8backend.entity.Follow;
import com.example.g8backend.entity.Message;
import com.example.g8backend.entity.User;
import com.example.g8backend.mapper.FollowMapper;
import com.example.g8backend.mapper.MessageMapper;
import com.example.g8backend.mapper.UserMapper;
import com.example.g8backend.service.impl.UserServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.test.util.ReflectionTestUtils;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.Mockito.*;

import static org.hamcrest.MatcherAssert.assertThat;

@ExtendWith(MockitoExtension.class)
public class UserServiceImplTest {

    @Mock
    private UserMapper userMapper;

    @Mock
    private FollowMapper followMapper;

    @Mock
    private MessageMapper messageMapper;

    @InjectMocks
    private UserServiceImpl userService;

    private User user1;
    private User user2;
    private Follow follow;
    private Message message;

    @BeforeEach
    void setUp() {
        user1 = new User().setUserId(1L).setUserName("user1");
        user2 = new User().setUserId(2L).setUserName("user2");
        ReflectionTestUtils.setField(userService, "baseMapper", userMapper);

        follow = new Follow().setFollowerId(1L).setFollowedId(2L);
        message = new Message()
                .setMessageId(100L)
                .setSenderId(1L)
                .setReceiverId(2L)
                .setContent("Hello")
                .setIsRead(false)
                .setSentAt(LocalDateTime.now());
    }

    // ------------------------- 关注功能测试 -------------------------
    @Test
    void followUser_Success() {
        // 模拟Mapper行为
        when(followMapper.insert(any(Follow.class))).thenReturn(1);

        // 测试关注
        boolean result = userService.followUser(1L, 2L);
        assertTrue(result);
        verify(followMapper, times(1)).insert(any(Follow.class));
    }

    @Test
    void followUser_SelfFollow_Fail() {
        // 尝试关注自己
        boolean result = userService.followUser(1L, 1L);
        assertFalse(result);
        verify(followMapper, never()).insert((Follow) any());
    }

    @Test
    void unfollowUser_Success() {
        // 模拟Mapper行为
        when(followMapper.deleteByPair(1L, 2L)).thenReturn(1);

        // 测试取消关注
        boolean result = userService.unfollowUser(1L, 2L);
        assertTrue(result);
        verify(followMapper, times(1)).deleteByPair(1L, 2L);
    }


    @Test
    void getFollowers_EmptyList() {
        // 模拟无粉丝
        when(followMapper.selectFollowers(1L)).thenReturn(Collections.emptyList());

        // 测试获取粉丝列表
        List<User> followers = userService.getFollowers(1L);
        assertTrue(followers.isEmpty());
    }

    // ------------------------- 私信功能测试 -------------------------
    @Test
    void sendMessage_Success() {
        // 模拟用户存在
        when(userMapper.selectById(2L)).thenReturn(user2);
        when(messageMapper.insert(any(Message.class))).thenAnswer(invocation -> {
            Message msg = invocation.getArgument(0);
            msg.setMessageId(100L);
            return 1;
        });

        // 测试发送消息
        Long messageId = userService.sendMessage(1L, 2L, "Hello");
        assertNotNull(messageId);
        verify(messageMapper, times(1)).insert(any(Message.class));
    }

    @Test
    void sendMessage_ReceiverNotExist_ThrowException() {
        // 模拟用户不存在
        when(userMapper.selectById(99L)).thenReturn(null);

        // 测试异常
        assertThrows(RuntimeException.class, () ->
                userService.sendMessage(1L, 99L, "Hello")
        );
        verify(messageMapper, never()).insert((Message) any());
    }

    @Test
    void getMessageHistory_SortedByTime() {
        // 模拟历史消息
        Message oldMessage = message.setSentAt(LocalDateTime.now().minusHours(1));
        when(messageMapper.selectUserMessages(1L))
                .thenReturn(Arrays.asList(oldMessage, message));

        // 测试按时间倒序排序
        List<Message> history = userService.getMessageHistory(1L);
        assertEquals(message.getMessageId(), history.get(0).getMessageId());
    }
}