package com.example.g8backend.controller;

import com.example.g8backend.dto.ApiResponse;
import com.example.g8backend.dto.UserRegisterDTO;
import com.example.g8backend.entity.User;
import com.example.g8backend.entity.UserStats;
import com.example.g8backend.service.IUserService;
import com.example.g8backend.service.IUserStatsService;
import com.example.g8backend.util.JwtUtil;
import com.example.g8backend.util.mailUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.web.bind.annotation.*;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@RestController
@RequestMapping("/auth")
public class AuthController {

    @Autowired
    private IUserService userService;

    @Autowired
    private IUserStatsService userStatsService;

    @Autowired
    private mailUtil mailUtil;

    @Autowired
    private PasswordEncoder passwordEncoder;

    @Autowired
    private JwtUtil jwtUtil;

    @Autowired
    private RedisTemplate<String, Object> redisTemplate;

    // 发送验证码
    @PostMapping("/send_verification_code")
    public ApiResponse<String> sendVerificationCode(@RequestBody UserRegisterDTO registerDTO) {
        if (userService.getUserByEmail(registerDTO.getEmail()) != null) {
            return ApiResponse.error(400, "邮箱已存在");
        }

        String verificationCode = UUID.randomUUID().toString().substring(0, 6);
        mailUtil.sendMail(registerDTO.getEmail(), "PT平台注册验证码", "您的验证码为：" + verificationCode + "，验证码十分钟内有效，请勿泄露。");

        redisTemplate.opsForValue().set(registerDTO.getEmail(), verificationCode, 10 * 60, java.util.concurrent.TimeUnit.SECONDS);
        return ApiResponse.success("验证码发送成功");
    }

    // 用户注册
    @PostMapping("/register")
    public ApiResponse<String> register(@RequestBody UserRegisterDTO registerDTO) {
        if (userService.getUserByName(registerDTO.getUserName()) != null) {
            return ApiResponse.error(400, "用户名已存在");
        }

        if (!redisTemplate.hasKey(registerDTO.getInvitationCode())) {
            return ApiResponse.error(400, "邀请码错误");
        }

        Object cachedCode = redisTemplate.opsForValue().get(registerDTO.getEmail());
        if (!registerDTO.getVerificationCode().equals(cachedCode)) {
            return ApiResponse.error(400, "验证码错误");
        }

        redisTemplate.delete(registerDTO.getEmail());

        User user = new User();
        user.setUserName(registerDTO.getUserName());
        user.setPassword(passwordEncoder.encode(registerDTO.getPassword()));
        user.setEmail(registerDTO.getEmail());
        user.setPasskey(UUID.randomUUID().toString().replace("-", ""));
        userService.save(user);

        UserStats userStats = new UserStats();
        userStats.setUserId(user.getUserId());
        userStats.setPasskey(user.getPasskey());
        userStatsService.save(userStats);

        return ApiResponse.message("注册成功");
    }

    // 用户登录
    @PostMapping("/login")
    public ApiResponse<Map<String, String>> login(@RequestBody User user) {
        User existingUser = userService.getUserByEmail(user.getEmail());
        if (existingUser == null || !passwordEncoder.matches(user.getPassword(), existingUser.getPassword())) {
            return ApiResponse.error(400, "用户名或密码错误");
        }

        String token = jwtUtil.generateToken(existingUser.getUserId());
        Map<String, String> response = new HashMap<>();
        response.put("token", token);

        return ApiResponse.success(response);
    }

    // 测试 Redis
    @GetMapping("/test_redis")
    public ApiResponse<String> testRedis() {
        Object value = redisTemplate.opsForValue().get("test");
        return ApiResponse.success("test redis ok");
    }
}
