package com.example.g8backend.service;

import com.example.g8backend.entity.PostRating;
import com.example.g8backend.mapper.PostMapper;
import com.example.g8backend.mapper.PostRatingMapper;
import com.example.g8backend.service.impl.PostRatingServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.transaction.annotation.Transactional;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
@Transactional
public class PostRatingServiceImplTest {

    @Mock
    private PostRatingMapper postRatingMapper;

    @Mock
    private PostMapper postMapper;

    @InjectMocks
    private PostRatingServiceImpl postRatingService;

    private final Long userId = 1L;
    private final Long postId = 100L;
    private final Integer validRating = 4;
    private final Integer invalidRating = 6;

    // 测试：合法评分应成功
    @Test
    public void testRatePost_Success() {
        // 模拟依赖行为
        when(postRatingMapper.insertOrUpdate(any(PostRating.class))).thenReturn(true);
        when(postRatingMapper.calculateAverageRating(postId)).thenReturn(4.0);
        when(postRatingMapper.getRatingCount(postId)).thenReturn(1);

        // 调用方法并验证无异常
        assertDoesNotThrow(() -> {
            postRatingService.ratePost(userId, postId, validRating);
        });

        // 验证数据库交互
        verify(postMapper).updateRatingStats(eq(postId), eq(4.0), eq(1));
    }

    // 测试：非法评分应抛出异常
    @Test
    public void testRatePost_InvalidRating() {
        // 调用方法并验证异常
        IllegalArgumentException exception = assertThrows(
                IllegalArgumentException.class,
                () -> postRatingService.ratePost(userId, postId, invalidRating)
        );
        assertEquals("评分值必须在1到5之间", exception.getMessage());

        // 验证未调用数据库操作
        verifyNoInteractions(postRatingMapper);
    }

    // 测试：重复评分应更新记录
    @Test
    public void testRatePost_UpdateExistingRating() {
        // 模拟已存在评分
        when(postRatingMapper.insertOrUpdate(any(PostRating.class))).thenReturn(true);
        when(postRatingMapper.calculateAverageRating(postId)).thenReturn(3.5, 4.0); // 两次调用返回不同值
        when(postRatingMapper.getRatingCount(postId)).thenReturn(1);

        // 同一用户对同一帖子二次评分
        assertDoesNotThrow(() -> {
            postRatingService.ratePost(userId, postId, 3);
            postRatingService.ratePost(userId, postId, 4);
        });

        // 验证两次更新统计信息
        verify(postMapper, times(2)).updateRatingStats(eq(postId), anyDouble(), eq(1));
    }

    // 测试：数据库操作失败应抛出异常
    @Test
    public void testRatePost_DatabaseFailure() {
        when(postRatingMapper.insertOrUpdate(any(PostRating.class))).thenReturn(false);
        RuntimeException exception = assertThrows(
                RuntimeException.class,
                () -> postRatingService.ratePost(userId, postId, validRating)
        );
        assertEquals("评分操作失败", exception.getMessage());
    }
    // 测试：获取评分用户数量
    @Test
    public void testGetRatingUserCount() {
        when(postRatingMapper.selectRatingUserCount(postId)).thenReturn(5L);
        Long count = postRatingService.getRatingUserCount(postId);
        assertEquals(5L, count);
    }
}