package com.example.g8backend.service;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.example.g8backend.entity.Report;
import com.example.g8backend.entity.User;
import com.example.g8backend.mapper.ReportMapper;
import com.example.g8backend.mapper.UserMapper;
import com.example.g8backend.service.impl.ReportServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import java.lang.reflect.Field;
import org.springframework.security.access.AccessDeniedException;

import java.lang.reflect.Method;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class ReportServiceImplTest {

    @Mock
    private ReportMapper reportMapper;

    @Mock
    private UserMapper userMapper;

    @InjectMocks
    private ReportServiceImpl reportService;

    private static final Long TEST_USER_ID = 1L;
    private static final Long TEST_ADMIN_ID = 999L;
    private static final Long TEST_REPORT_ID = 1000L;

    private User adminUser;
    private User normalUser;
    private Report pendingReport;

    @BeforeEach
    void setUp() {
        // 初始化管理员用户
        adminUser = new User();
        adminUser.setUserId(TEST_ADMIN_ID);
        adminUser.setRole("ADMIN");

        // 初始化普通用户
        normalUser = new User();
        normalUser.setUserId(TEST_USER_ID);
        normalUser.setRole("USER");

        // 初始化待处理举报
        pendingReport = new Report();
        pendingReport.setReportId(TEST_REPORT_ID);
        pendingReport.setStatus("pending");
    }

    // ------------------------- submitReport 测试 -------------------------
    @Test
    void submitReport_WhenNoDuplicate_ShouldReturnTrue() {
        when(reportMapper.selectCount(any())).thenReturn(0L);
        when(reportMapper.insert(any(Report.class))).thenReturn(1);

        boolean result = reportService.submitReport(TEST_USER_ID, 2L, "违规内容");
        assertTrue(result);
        verify(reportMapper).insert(any(Report.class));
    }

    // ------------------------- resolveReport 测试 -------------------------
    @Test
    void resolveReport_WhenAdminProcesses_ShouldSucceed() {
        when(userMapper.selectById(TEST_ADMIN_ID)).thenReturn(adminUser);
        Report mockReport = new Report();
        mockReport.setStatus("pending");
        when(reportMapper.selectById(TEST_REPORT_ID)).thenReturn(mockReport);

        when(reportMapper.updateById(any(Report.class))).thenReturn(1);

        boolean result = reportService.resolveReport(TEST_REPORT_ID, TEST_ADMIN_ID, "resolved", "已处理");

        assertTrue(result);
        verify(reportMapper).updateById(any(Report.class)); // 明确验证参数类型
    }

    @Test
    void resolveReport_WhenNonAdminAttempts_ShouldThrowAccessDenied() {
        when(userMapper.selectById(TEST_USER_ID)).thenReturn(normalUser);

        // 使用正确的异常类型
        assertThrows(AccessDeniedException.class, () ->
                reportService.resolveReport(TEST_REPORT_ID, TEST_USER_ID, "resolved", null)
        );
        verify(reportMapper, never()).updateById(any(Report.class));
    }
    // ------------------------- getReports 测试 -------------------------
    @Test
    void getReports_WhenAdminQueries_ShouldReturnFiltered() {
        // 模拟管理员存在
        when(userMapper.selectById(TEST_ADMIN_ID)).thenReturn(adminUser);
        when(reportMapper.selectList(any())).thenReturn(Collections.singletonList(pendingReport));

        // 执行方法
        List<Report> result = reportService.getReports("pending", TEST_ADMIN_ID);
        assertEquals(1, result.size());

        // 验证是否调用了 selectList 方法
        verify(reportMapper).selectList(any(QueryWrapper.class));

        // 打印实际 SQL 用于调试（可选）
        QueryWrapper<Report> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("status", "pending");
        System.out.println("Expected SQL: " + queryWrapper.getTargetSql());
    }
    @Test
    void getReports_WhenNonAdminQueries_ShouldThrowAccessDenied() {
        when(userMapper.selectById(TEST_USER_ID)).thenReturn(normalUser);
        assertThrows(AccessDeniedException.class, () ->
                reportService.getReports("pending", TEST_USER_ID)
        );
        verify(reportMapper, never()).selectList(any());
    }
}