package com.example.g8backend.service;

import com.example.g8backend.entity.Post;
import com.example.g8backend.mapper.PostMapper;
import com.example.g8backend.service.impl.PostServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.junit.jupiter.SpringExtension;

import java.sql.Timestamp;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

@ExtendWith(SpringExtension.class)
@DisplayName("帖子服务测试")
class PostServiceTest {

    @Mock
    private PostMapper postMapper;

    private PostServiceImpl postService;

    private Post testPost;

    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
        postService = new PostServiceImpl(postMapper);
        testPost = createTestPost();
    }

    private Post createTestPost() {
        Post post = new Post();
        post.setPostId(1L);
        post.setUserId(1L);
        post.setPostTitle("测试标题");
        post.setPostContent("测试内容");
        post.setCreatedAt(new Timestamp(System.currentTimeMillis()));
        return post;
    }

    @Test
    @DisplayName("创建帖子-成功")
    void save_ShouldSucceed() {
        // Arrange
        when(postMapper.insert(any(Post.class))).thenReturn(1);

        // Act
        boolean result = postService.save(testPost);

        // Assert
        assertTrue(result);
        verify(postMapper).insert(testPost);
    }

    @Test
    @DisplayName("获取帖子-通过ID存在")
    void getById_WhenExists_ShouldReturnPost() {
        // Arrange
        when(postMapper.selectById(1L)).thenReturn(testPost);

        // Act
        Post result = postService.getById(1L);

        // Assert
        assertNotNull(result);
        assertEquals(testPost.getPostId(), result.getPostId());
        verify(postMapper).selectById(1L);
    }

    @Test
    @DisplayName("获取帖子-通过ID不存在")
    void getById_WhenNotExists_ShouldReturnNull() {
        // Arrange
        when(postMapper.selectById(999L)).thenReturn(null);

        // Act
        Post result = postService.getById(999L);

        // Assert
        assertNull(result);
        verify(postMapper).selectById(999L);
    }

    @Test
    @DisplayName("更新帖子-成功")
    void updateById_ShouldSucceed() {
        // Arrange
        when(postMapper.updateById(any(Post.class))).thenReturn(1);

        // Act
        boolean result = postService.updateById(testPost);

        // Assert
        assertTrue(result);
        verify(postMapper).updateById(testPost);
    }

    @Test
    @DisplayName("删除帖子-成功")
    void removeById_ShouldSucceed() {
        // Arrange
        when(postMapper.deleteById(1L)).thenReturn(1);

        // Act
        boolean result = postService.removeById(1L);

        // Assert
        assertTrue(result);
        verify(postMapper).deleteById(1L);
    }

    @Test
    @DisplayName("获取用户帖子列表")
    void getPostsByUserId_ShouldReturnPosts() {
        // Arrange
        List<Post> expectedPosts = Arrays.asList(testPost);
        when(postMapper.getPostsByUserId(1L)).thenReturn(expectedPosts);

        // Act
        List<Post> result = postService.getPostsByUserId(1L);

        // Assert
        assertNotNull(result);
        assertFalse(result.isEmpty());
        assertEquals(testPost.getPostId(), result.get(0).getPostId());
        verify(postMapper).getPostsByUserId(1L);
    }

    @Test
    @DisplayName("获取用户帖子-空列表")
    void getPostsByUserId_WhenNoPosts_ShouldReturnEmptyList() {
        // Arrange
        when(postMapper.getPostsByUserId(999L)).thenReturn(Arrays.asList());

        // Act
        List<Post> result = postService.getPostsByUserId(999L);

        // Assert
        assertNotNull(result);
        assertTrue(result.isEmpty());
        verify(postMapper).getPostsByUserId(999L);
    }
}