package com.example.g8backend.util;

import com.dampcake.bencode.Bencode;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.security.MessageDigest;
import static org.junit.jupiter.api.Assertions.*;

class TorrentUtilTest {

    private static final String TEST_TRACKER_URL = "http://test.tracker/announce";

    @TempDir
    Path tempDir; // 临时目录

    //---------------- 动态生成 .torrent 文件 -----------------
    private File createTestTorrentFile(String announceUrl, Map<String, Object> info) throws Exception {
        // 1. 构造 Bencode 数据结构
        Map<String, Object> torrentMap = new HashMap<>();
        torrentMap.put("announce", announceUrl);
        torrentMap.put("info", info);

        // 2. 编码为 Bencode 字节流
        Bencode bencode = new Bencode();
        byte[] bencodeData = bencode.encode(torrentMap);

        // 3. 写入临时文件
        Path torrentPath = tempDir.resolve("dynamic.torrent");
        Files.write(torrentPath, bencodeData);
        return torrentPath.toFile();
    }

    //---------------- 测试 injectTracker() -----------------
    @Test
    void testInjectTracker() throws Exception {
        // 动态生成测试文件
        Map<String, Object> info = Map.of("name", "test.txt", "length", 1024);
        File originalFile = createTestTorrentFile("http://old.tracker", info);

        // 调用方法修改 Tracker
        byte[] modifiedData = TorrentUtil.injectTracker(originalFile, TEST_TRACKER_URL);

        // 验证新 Tracker 是否生效
        String modifiedBencode = new String(modifiedData);
        assertTrue(modifiedBencode.contains(TEST_TRACKER_URL));
    }

    //---------------- 测试 getInfoHash() -----------------
    @Test
    void testGetInfoHash_ShouldCalculateCorrectHash() throws Exception {
        // 动态生成测试文件
        Map<String, Object> info = Map.of(
                "name", "test.txt",
                "length", 1024,
                "piece length", 16384,
                "pieces", "1234567890abcdef1234"
        );
        File torrentFile = createTestTorrentFile(TEST_TRACKER_URL, info);

        // 计算预期哈希（直接通过 Bencode 库生成 info 的哈希）
        Bencode bencode = new Bencode();
        byte[] infoBytes = bencode.encode(info);
        String expectedHash = TorrentUtil.bytesToHex(
                MessageDigest.getInstance("SHA-1").digest(infoBytes)
        );

        // 调用方法并验证
        String actualHash = TorrentUtil.getInfoHash(torrentFile);
        assertEquals(expectedHash, actualHash);
    }

    //---------------- 测试 saveToFile() -----------------
    @Test
    void testSaveToFile() throws Exception {
        // 生成测试数据
        byte[] testData = {0x01, 0x02, 0x03};
        File outputFile = tempDir.resolve("output.torrent").toFile();

        // 调用方法保存文件
        TorrentUtil.saveToFile(testData, outputFile);

        // 验证文件内容
        byte[] savedData = Files.readAllBytes(outputFile.toPath());
        assertArrayEquals(testData, savedData);
    }
      //---------------- 异常场景测试 -----------------
    @Test
    void getInfoHash_ShouldThrow_WhenInfoFieldMissing() throws Exception {
        // 构造一个缺少 "info" 字段的无效 .torrent 文件
        Map<String, Object> invalidTorrent = Map.of("announce", TEST_TRACKER_URL);
        Bencode bencode = new Bencode();
        byte[] bencodeData = bencode.encode(invalidTorrent);

        Path invalidTorrentPath = tempDir.resolve("dynamic.torrent");
        Files.write(invalidTorrentPath, bencodeData);

        // 验证抛出异常
        assertThrows(IllegalArgumentException.class, () -> TorrentUtil.getInfoHash(invalidTorrentPath.toFile()));
    }
}