package com.example.g8backend.service;

import com.example.g8backend.entity.Post;
import com.example.g8backend.entity.User;
import com.example.g8backend.mapper.UserMapper;
import com.example.g8backend.service.impl.AdminServiceImpl;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import com.example.g8backend.mapper.PostMapper;

import java.time.LocalDateTime;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class AdminServiceImplTest {
    @Mock
    private UserMapper userMapper;
    @InjectMocks
    private AdminServiceImpl adminService;
    @Mock
    private PostMapper postMapper;

    @Test
    public void testGrantVip_Success() {
        User user = new User();
        user.setUserId(1L);
        when(userMapper.selectById(1L)).thenReturn(user);

        boolean result = adminService.grantVip(1L);
        assertTrue(result);
        assertEquals("vip", user.getUserLevel());
    }

    @Test
    public void testGrantVip_UserNotFound() {
        when(userMapper.selectById(1L)).thenReturn(null);
        boolean result = adminService.grantVip(1L);
        assertFalse(result);
    }

    void testBanUser_Success() {
        User user = new User().setUserId(1L);
        when(userMapper.selectById(1L)).thenReturn(user);
        when(userMapper.updateById(any(User.class))).thenReturn(1);

        boolean result = adminService.banUser(1L, "违规操作", 1001L);
        assertTrue(result);

        assertTrue(user.getIsBanned());
        assertEquals("违规操作", user.getBannedReason());
        assertNotNull(user.getBannedAt());
        assertEquals(1001L, user.getBannedBy());
        verify(userMapper).updateById(any(User.class));
    }

    @Test
    void testBanUser_UserNotFound() {
        when(userMapper.selectById(1L)).thenReturn(null);
        boolean result = adminService.banUser(1L, "原因", 1001L);
        assertFalse(result);
        verify(userMapper, never()).updateById(any(User.class));
    }

    @Test
    void testUnbanUser_Success() {
        User user = new User().setUserId(1L)
                .setIsBanned(true)
                .setBannedReason("原因")
                .setBannedAt(LocalDateTime.now())
                .setBannedBy(1001L);
        when(userMapper.selectById(1L)).thenReturn(user);
        when(userMapper.updateById(any(User.class))).thenReturn(1);

        boolean result = adminService.unbanUser(1L, 1001L);
        assertTrue(result);

        assertFalse(user.getIsBanned());
        assertNull(user.getBannedReason());
        assertNull(user.getBannedAt());
        assertNull(user.getBannedBy());
        verify(userMapper).updateById(any(User.class));
    }

    @Test
    void testUnbanUser_UserNotFound() {
        when(userMapper.selectById(1L)).thenReturn(null);
        boolean result = adminService.unbanUser(1L, 1001L);
        assertFalse(result);
        verify(userMapper, never()).updateById(any(User.class));
    }

    @Test
    void testLockPost_Success() {
        Post post = new Post().setPostId(1L);
        when(postMapper.selectById(1L)).thenReturn(post);
        when(postMapper.updateById(any(Post.class))).thenReturn(1);

        boolean result = adminService.lockPost(1L, "违规内容", 1001L);
        assertTrue(result);

        assertTrue(post.getIsLocked());
        assertEquals("违规内容", post.getLockedReason());
        assertNotNull(post.getLockedAt());
        assertEquals(1001L, post.getLockedBy());
        verify(postMapper).updateById(any(Post.class));
    }

    @Test
    void testLockPost_PostNotFound() {
        when(postMapper.selectById(1L)).thenReturn(null);
        boolean result = adminService.lockPost(1L, "原因", 1001L);
        assertFalse(result);
        verify(postMapper, never()).updateById(any(Post.class));
    }

    @Test
    void testUnlockPost_Success() {
        Post post = new Post().setPostId(1L)
                .setIsLocked(true)
                .setLockedReason("原因")
                .setLockedAt(LocalDateTime.now())
                .setLockedBy(1001L);
        when(postMapper.selectById(1L)).thenReturn(post);
        when(postMapper.updateById(any(Post.class))).thenReturn(1);

        boolean result = adminService.unlockPost(1L, 1001L);
        assertTrue(result);

        assertFalse(post.getIsLocked());
        assertNull(post.getLockedReason());
        assertNull(post.getLockedAt());
        assertNull(post.getLockedBy());
        verify(postMapper).updateById(any(Post.class));
    }
    @Test
    void testUnlockPost_PostNotFound() {
        when(postMapper.selectById(1L)).thenReturn(null);
        boolean result = adminService.unlockPost(1L, 1001L);
        assertFalse(result);
        verify(postMapper, never()).updateById(any(Post.class));
    }
}