package com.example.g8backend.service;

import com.dampcake.bencode.Bencode;
import com.example.g8backend.entity.Torrent;
import com.example.g8backend.mapper.TorrentMapper;
import com.example.g8backend.service.impl.TorrentServiceImpl;

import com.example.g8backend.util.TorrentUtil;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.junit.jupiter.api.io.TempDir;
import org.mockito.*;
import org.mockito.junit.jupiter.MockitoExtension;

import java.io.File;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class TorrentServiceTest {
    @InjectMocks
    private TorrentServiceImpl torrentService;

    @Mock
    private TorrentMapper torrentMapper;

    @TempDir
    Path tempDir;

    private File createTestTorrentFile(Map<String, Object> info) throws Exception {
        // 1. 构造 Bencode 数据结构
        Map<String, Object> torrentMap = new HashMap<>();
        torrentMap.put("announce", "http://localhost:6881/announce");
        torrentMap.put("info", info);

        // 2. 编码为 Bencode 字节流
        Bencode bencode = new Bencode();
        byte[] bencodeData = bencode.encode(torrentMap);

        // 3. 写入临时文件
        Path torrentPath = tempDir.resolve("dynamic.torrent");
        Files.write(torrentPath, bencodeData);
        return torrentPath.toFile();
    }

    @BeforeEach
    public void setUp() {
        torrentService = new TorrentServiceImpl();
        torrentMapper = Mockito.mock(TorrentMapper.class);
        // 注入 Mock 对象（反射）
        try {
            var field = TorrentServiceImpl.class.getDeclaredField("torrentMapper");
            field.setAccessible(true);
            field.set(torrentService, torrentMapper);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Test
    public void testHandleTorrentUpload() throws Exception {
        Map<String, Object> info = Map.of("name", "test.txt", "length", 1024);
        File torrentFile = createTestTorrentFile(info);
        Long userId = 1L;
        String passkey = "123456";
        byte[] mockedBytes = "modified".getBytes();
        String expectedInfoHash = "modified-info-hash";

        try (MockedStatic<TorrentUtil> mockedStatic = Mockito.mockStatic(TorrentUtil.class)) {
            mockedStatic.when(() -> TorrentUtil.injectTracker(any(File.class), any(String.class)))
                    .thenReturn(mockedBytes);
            mockedStatic.when(() -> TorrentUtil.getInfoHash(any(File.class)))
                    .thenReturn(expectedInfoHash);

            Torrent torrent = torrentService.handleTorrentUpload(torrentFile, "test.torrent", userId, passkey);

            // 使用 ArgumentCaptor 捕获 insert 方法的参数
            ArgumentCaptor<Torrent> torrentCaptor = ArgumentCaptor.forClass(Torrent.class);
            verify(torrentMapper, times(1)).insert(torrentCaptor.capture());
            Torrent captured = torrentCaptor.getValue();

            assertEquals(userId, captured.getUserId());
            assertEquals("test.torrent", captured.getTorrentName());
            assertEquals(expectedInfoHash, captured.getInfoHash());
            assertEquals(torrentFile.length() / 1024.0 / 1024.0, captured.getFileSize());

            assertEquals(expectedInfoHash, torrent.getInfoHash());
            assertEquals(userId, torrent.getUserId());
            assertEquals(torrentFile.length() / 1024.0 / 1024.0, torrent.getFileSize());
        } finally {
            if (!torrentFile.delete()) {
                System.err.println("Failed to delete temporary file: " + torrentFile.getAbsolutePath());
            }
        }
    }


    @Test
    public void testHandleTorrentDownload() throws Exception {
        Torrent torrent = new Torrent();
        torrent.setTorrentName("sample.torrent");

        File fakeTorrentFile = new File("uploaded-torrents/sample.torrent");
        fakeTorrentFile.getParentFile().mkdirs();
        byte[] originalBytes = "original-data".getBytes();
        Files.write(fakeTorrentFile.toPath(), originalBytes);

        String passkey = "testpasskey";
        byte[] modifiedBytes = "modified-data".getBytes();

        try (MockedStatic<TorrentUtil> mockedStatic = Mockito.mockStatic(TorrentUtil.class)) {
            // mock injectTracker
            mockedStatic.when(() -> TorrentUtil.injectTracker(any(File.class), anyString()))
                    .thenReturn(modifiedBytes);

            File downloaded = torrentService.handleTorrentDownload(torrent, passkey);

            assertTrue(downloaded.exists());
            assertEquals("modified-data", Files.readString(downloaded.toPath()));

            // 清理临时文件
            downloaded.delete();
        } finally {
            fakeTorrentFile.delete();  // 清理原始文件
        }
    }


    @Test
    public void testFindByInfoHash() {
        Torrent mockTorrent = new Torrent();
        mockTorrent.setInfoHash("abc123");
        when(torrentMapper.getTorrentByInfoHash("abc123")).thenReturn(mockTorrent);

        Torrent result = torrentService.findByInfoHash("abc123");

        assertEquals("abc123", result.getInfoHash());
        verify(torrentMapper).getTorrentByInfoHash("abc123");
    }

    @Test
    public void testFindByTorrentId() {
        Torrent mockTorrent = new Torrent();
        mockTorrent.setTorrentId(42L);
        when(torrentMapper.getTorrentByTorrentId(42L)).thenReturn(mockTorrent);

        Torrent result = torrentService.findByTorrentId(42L);

        assertEquals(42L, result.getTorrentId());
        verify(torrentMapper).getTorrentByTorrentId(42L);
    }
}