package com.example.g8backend.service.impl;

import com.example.g8backend.dto.TorrentRecommendationDTO;
import com.example.g8backend.entity.Torrent;
import com.example.g8backend.mapper.PeerMapper;
import com.example.g8backend.mapper.TorrentMapper;
import com.example.g8backend.service.ITorrentRecommendationService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

@Service
public class TorrentRecommendationServiceImpl implements ITorrentRecommendationService {

    private final PeerMapper peerMapper;
    private final TorrentMapper torrentMapper;

    @Autowired
    public TorrentRecommendationServiceImpl(PeerMapper peerMapper, TorrentMapper torrentMapper) {
        this.peerMapper = peerMapper;
        this.torrentMapper = torrentMapper;
    }

    /**
     * 获取冷门种子推荐列表，冷门定义为“种子存在时间越长，最近7天内peer活跃数越少”
     *
     * @param limit 返回结果条数限制
     * @return 冷门种子推荐DTO列表
     */
    @Override
    @Transactional(readOnly = true)
    public List<TorrentRecommendationDTO> getColdTorrentRecommendations(int limit) {
        // 1. 查询所有有peer活动的种子info_hash列表
        List<String> infoHashes = peerMapper.selectAllInfoHashesWithPeers();
        if (infoHashes == null || infoHashes.isEmpty()) {
            return new ArrayList<>();
        }

        // 2. 根据info_hash列表查询对应的种子详细信息
        List<Torrent> allTorrents = torrentMapper.selectByInfoHashList(infoHashes);

        LocalDateTime now = LocalDateTime.now();
        List<TorrentRecommendationDTO> dtos = new ArrayList<>();

        // 3. 计算每个种子的冷度指标
        for (Torrent torrent : allTorrents) {
            LocalDateTime uploadTime = torrent.getUploadTime();
            if (uploadTime == null) {
                // 如果没有上传时间，跳过
                continue;
            }

            // 计算种子发布距今的小时数
            long ageHours = Duration.between(uploadTime, now).toHours();

            // 查询最近7天内该infoHash的peer活跃数
            Long recentActivityCount = peerMapper.countRecentActivity(torrent.getInfoHash());
            if (recentActivityCount == null) {
                recentActivityCount = 0L;
            }

            // 计算冷度分，防止除0，加1
            double coldnessScore = (double) ageHours / (recentActivityCount + 1);

            // 组装DTO
            TorrentRecommendationDTO dto = new TorrentRecommendationDTO();
            dto.setTorrentId(torrent.getTorrentId());
            dto.setTorrentName(torrent.getTorrentName());
            dto.setInfoHash(torrent.getInfoHash());
            dto.setFileSize(torrent.getFileSize());
            dto.setUploadTime(torrent.getUploadTime());
            dto.setAgeHours(ageHours);
            dto.setRecentActivityCount(recentActivityCount);
            dto.setColdnessScore(coldnessScore);

            dtos.add(dto);
        }

        // 4. 按冷度分倒序排序，取前limit条
        dtos.sort(Comparator.comparingDouble(TorrentRecommendationDTO::getColdnessScore).reversed());
        if (dtos.size() > limit) {
            return dtos.subList(0, limit);
        }
        return dtos;
    }

    @Override
    @Transactional
    public void refreshIsRareField(int limit) {
        // 1. 获取所有种子的冷门推荐（冷门度排序）
        List<TorrentRecommendationDTO> coldTorrents = getColdTorrentRecommendations(Integer.MAX_VALUE);

        // 2. 前limit条设为 israre = 1，其他设为0
        for (int i = 0; i < coldTorrents.size(); i++) {
            TorrentRecommendationDTO dto = coldTorrents.get(i);
            int israre = (i < limit) ? 1 : 0;

            // 直接调用mapper更新对应torrent的israre字段
            torrentMapper.updateIsRareByInfoHash(dto.getInfoHash(), israre);
        }
    }
}
