package com.example.g8backend.service.impl;

import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.example.g8backend.entity.Follow;
import com.example.g8backend.entity.Message;
import com.example.g8backend.entity.User;
import com.example.g8backend.mapper.FollowMapper;
import com.example.g8backend.mapper.MessageMapper;
import com.example.g8backend.mapper.UserMapper;
import com.example.g8backend.service.IUserService;
import jakarta.annotation.Resource;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.stream.Collectors;


@Service
public class UserServiceImpl extends ServiceImpl<UserMapper, User> implements IUserService {
    @Resource
    private UserMapper userMapper; // 手动注入 UserMapper

    @Override
    public User getUserByName(String name) { return userMapper.getUserByName(name);} // 调用 UserMapper 的自定义 SQL

    @Override
    public User getUserByEmail(String email) { return userMapper.getUserByEmail(email);}

    @Override
    public User getUserByPasskey(String passkey) { return userMapper.getUserByPasskey(passkey);}

    @Resource
    private FollowMapper followMapper;
    @Resource
    private MessageMapper messageMapper;

    @Override
    public boolean followUser(Long followerId, Long followedId) {
        if (followerId.equals(followedId)) return false;
        Follow follow = new Follow();
        follow.setFollowerId(followerId);
        follow.setFollowedId(followedId);
        return followMapper.insert(follow) > 0;
    }

    @Override
    public boolean unfollowUser(Long followerId, Long followedId) {
        // 删除关注关系
        return followMapper.deleteByPair(followerId, followedId) > 0;
    }

    @Override
    public List<User> getFollowings(Long userId) {
        // 1. 获取关注ID列表
        List<Long> followingIds = followMapper.selectFollowings(userId);
        // 2. 批量查询用户信息
        return followingIds.isEmpty() ?
                Collections.emptyList() :
                this.listByIds(followingIds).stream()
                        .peek(user -> user.setPassword(null)) // 敏感信息脱敏
                        .collect(Collectors.toList());
    }

    @Override
    public List<User> getFollowers(Long userId) {
        // 1. 获取粉丝ID列表
        List<Long> followerIds = followMapper.selectFollowers(userId);
        // 2. 批量查询用户信息
        return followerIds.isEmpty() ?
                Collections.emptyList() :
                this.listByIds(followerIds).stream()
                        .peek(user -> user.setPassword(null))
                        .collect(Collectors.toList());
    }

    @Override
    public Long sendMessage(Long senderId, Long receiverId, String content) {
        // 1. 校验接收者是否存在（方案一）
        if (userMapper.selectById(receiverId) == null) {
            throw new RuntimeException("接收用户不存在");
        }

        // 2. 创建消息实体
        Message message = new Message()
                .setSenderId(senderId)
                .setReceiverId(receiverId)
                .setContent(content)
                .setSentAt(LocalDateTime.now());

        // 3. 插入数据库
        messageMapper.insert(message);
        return message.getMessageId();
    }

    @Override
    public List<Message> getMessages(Long userId, Long partnerId) {
        // 获取双方对话记录
        return messageMapper.selectConversation(userId, partnerId).stream()
                .peek(msg -> {
                    // 标记消息为已读
                    if (!msg.getIsRead() && msg.getReceiverId().equals(userId)) {
                        msg.setIsRead(true);
                        messageMapper.updateById(msg);
                    }
                })
                .collect(Collectors.toList());
    }

    @Override
    public List<Message> getMessageHistory(Long userId) {
        // 获取用户所有相关消息
        return messageMapper.selectUserMessages(userId).stream()
                .sorted(Comparator.comparing(Message::getSentAt).reversed())
                .collect(Collectors.toList());
    }

}
