package com.example.g8backend.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.example.g8backend.dto.ApiResponse;
import com.example.g8backend.dto.PostCreateDTO;
import com.example.g8backend.dto.PostHistoryDTO;
import com.example.g8backend.entity.Post;
import com.example.g8backend.entity.PostView;
import com.example.g8backend.mapper.PostViewMapper;
import com.example.g8backend.service.IPostService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/post")
public class PostController {

    @Autowired
    private IPostService postService;

    @Autowired
    private PostViewMapper postViewMapper;

    @PostMapping("")
    public ResponseEntity<ApiResponse<Void>> createPost(@RequestBody PostCreateDTO postCreateDTO) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Post post = postCreateDTO.getPost();
        Long[] tagIds = postCreateDTO.getTagIds();
        post.setUserId(userId);
        if (tagIds.length > 0) {
            postService.createPost(post, tagIds);
        } else {
            postService.createPost(post);
        }
        return ResponseEntity.ok(ApiResponse.message("Post created successfully."));
    }

    @GetMapping("/{postId}")
    public ResponseEntity<ApiResponse<Post>> getPost(@PathVariable Long postId) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        postService.recordViewHistory(userId, postId);
        Post post = postService.getById(postId);
        return ResponseEntity.ok(ApiResponse.success(post));
    }

    @DeleteMapping("/{postId}")
    public ResponseEntity<ApiResponse<String>> deletePost(@PathVariable Long postId) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Post post = postService.getById(postId);
        if (post == null) {
            return ResponseEntity.status(404).body(ApiResponse.error(404, "Post not found."));
        }
        if (post.getUserId() != userId) {
            return ResponseEntity.status(403).body(ApiResponse.error(403, "You are not authorized to delete this post."));
        }
        postService.removeById(postId);
        return ResponseEntity.ok(ApiResponse.message("Post deleted successfully."));
    }

    @GetMapping("/getAll")
    public ResponseEntity<ApiResponse<List<Post>>> getAllPosts() {
        return ResponseEntity.ok(ApiResponse.success(postService.list()));
    }

    @GetMapping("/getByUserId/{userId}")
    public ResponseEntity<ApiResponse<List<Post>>> getPostsByUserId(@PathVariable Long userId) {
        return ResponseEntity.ok(ApiResponse.success(postService.getPostsByUserId(userId)));
    }

    @PutMapping("/{postId}")
    public ResponseEntity<ApiResponse<String>> updatePost(@PathVariable Long postId, @RequestBody Post post) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Post existingPost = postService.getById(postId);
        if (existingPost == null) {
            return ResponseEntity.status(404).body(ApiResponse.error(404, "Post not found."));
        }
        if (existingPost.getUserId() != userId) {
            return ResponseEntity.status(403).body(ApiResponse.error(403, "You are not authorized to update this post."));
        }

        post.setPostId(postId);
        post.setUserId(userId);
        postService.updateById(post);
        return ResponseEntity.ok(ApiResponse.message("Post updated successfully."));
    }

    @GetMapping("/type/{postType}")
    public ResponseEntity<ApiResponse<List<Post>>> getPostsByType(@PathVariable String postType) {
        List<Post> posts = postService.getPostsByType(postType);
        return ResponseEntity.ok(ApiResponse.success(posts));
    }

    @PostMapping("/{postId}/like")
    public ResponseEntity<ApiResponse<String>> likePost(@PathVariable Long postId) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        postService.likePost(userId, postId);
        return ResponseEntity.ok(ApiResponse.message("Post liked successfully."));
    }

    @DeleteMapping("/{postId}/like")
    public ResponseEntity<ApiResponse<String>> unlikePost(@PathVariable Long postId) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        postService.unlikePost(userId, postId);
        return ResponseEntity.ok(ApiResponse.message("Post unliked successfully."));
    }

    @GetMapping("/{postId}/likes")
    public ResponseEntity<ApiResponse<Long>> getPostLikeCount(@PathVariable Long postId) {
        Long likeCount = postService.getPostLikeCount(postId);
        return ResponseEntity.ok(ApiResponse.success(likeCount));
    }

    @GetMapping("/search")
    public ResponseEntity<ApiResponse<List<Post>>> searchPosts(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) List<Long> tags,
            @RequestParam(required = false) String author) {
        List<Post> result = postService.searchPosts(keyword, tags, author);
        return ResponseEntity.ok(ApiResponse.success(result));
    }

    @GetMapping("/history")
    public ResponseEntity<ApiResponse<List<PostHistoryDTO>>> getViewHistory() {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        List<PostHistoryDTO> history = postService.getViewHistoryWithTitles(userId);
        return ResponseEntity.ok(ApiResponse.success(history));
    }

    @GetMapping("/recommended")
    public ResponseEntity<ApiResponse<Page<Post>>> getRecommendedPosts(
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Page<Post> pageResult = postService.getRecommendedPosts(page, size, userId);
        return ResponseEntity.ok(ApiResponse.success(pageResult));
    }

    @GetMapping("/recommended-by-tags")
    public ResponseEntity<ApiResponse<Page<Post>>> getRecommendedByTags(
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Page<Post> result = postService.getRecommendedByTags(page, size, userId);
        return ResponseEntity.ok(ApiResponse.success(result));
    }
}
