package com.example.g8backend.controller;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.example.g8backend.dto.PostCreateDTO;
import com.example.g8backend.dto.PostHistoryDTO;
import com.example.g8backend.entity.PostView;
import com.example.g8backend.mapper.PostViewMapper;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;
import com.example.g8backend.entity.Post;
import com.example.g8backend.service.IPostService;
import java.util.List;
@RestController
@RequestMapping("/post")
public class PostController {
    @Autowired
    private IPostService postService;
    @Autowired  // ✅ 新增注入
    private PostViewMapper postViewMapper;
    @PostMapping("")
    public ResponseEntity<?> createPost(@RequestBody PostCreateDTO postCreateDTO) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        Post post = postCreateDTO.getPost();
        Long[] tagIds = postCreateDTO.getTagIds();
        post.setUserId(userId);
        if (tagIds.length > 0){
            postService.createPost(post, tagIds);
        } else {
            postService.createPost(post);
        }
        return ResponseEntity.ok().build();
    }
    @GetMapping("/{postId}")
    public Post getPost(@PathVariable Long postId) {
        // 获取当前用户ID
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        // 记录浏览行为
        postService.recordViewHistory(userId, postId);
        // 返回帖子详情
        return postService.getById(postId);
    }
    @DeleteMapping("/{postId}")
    public ResponseEntity<?> deletePost(@PathVariable("postId") Long postId) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        Post post = postService.getById(postId);
        if (post == null) {
            return ResponseEntity.status(500).body("Post not found.");
        }
        if (post.getUserId()!= userId) {
            return ResponseEntity.status(403).body("You are not authorized to delete this post.");
        }
        postService.removeById(postId);
        return ResponseEntity.ok().body("Post deleted successfully.");
    }
    @GetMapping("/getAll")
    public List<Post> getAllPosts() {
        return postService.list();
    }
    @GetMapping("/getByUserId/{userId}")
    public List<Post> getPostsByUserId(@PathVariable("userId") Long userId) {
        return postService.getPostsByUserId(userId);
    }
    @PutMapping("/{postId}")
    public ResponseEntity<?> updatePost(@PathVariable("postId") Long postId, @RequestBody Post post) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        Post existingPost = postService.getById(postId);

        if (existingPost == null) {
            return ResponseEntity.status(500).body("Post not found.");
        }
        if (existingPost.getUserId() != userId) {
            return ResponseEntity.status(403).body("You are not authorized to update this post.");
        }

        post.setPostId(postId);
        post.setUserId(userId);
        postService.updateById(post);
        return ResponseEntity.ok().body("Post updated successfully.");
    }
    @GetMapping("/type/{postType}")
    public ResponseEntity<?> getPostsByType(@PathVariable String postType) {
        List<Post> posts = postService.getPostsByType(postType);
        return ResponseEntity.ok().body(posts);
    }
    @PostMapping("/{postId}/like")
    public ResponseEntity<?> likePost(@PathVariable Long postId) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        postService.likePost(userId, postId);
        return ResponseEntity.ok().body("Post liked successfully.");
    }
    @DeleteMapping("/{postId}/like")
    public ResponseEntity<?> unlikePost(@PathVariable Long postId) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        postService.unlikePost(userId, postId);
        return ResponseEntity.ok().body("Post unliked successfully.");
    }
    @GetMapping("/{postId}/likes")
    public ResponseEntity<?> getPostLikeCount(@PathVariable Long postId) {
        Long likeCount = postService.getPostLikeCount(postId);
        return ResponseEntity.ok().body(likeCount);
    }
    // 搜索帖子
    @GetMapping("/search")
    public List<Post> searchPosts(
            @RequestParam(required = false) String keyword,
            @RequestParam(required = false) List<Long> tags,  // 修改为接收多个标签
            @RequestParam(required = false) String author) {
        return postService.searchPosts(keyword, tags, author);
    }

    @GetMapping("/history")
    public ResponseEntity<List<PostHistoryDTO>> getViewHistory() {
        // 获取当前用户ID
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();

        // 调用Service层
        List<PostHistoryDTO> history = postService.getViewHistoryWithTitles(userId);
        return ResponseEntity.ok(history);
    }
    @GetMapping("/recommended")
    public ResponseEntity<Page<Post>> getRecommendedPosts(
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        // 获取当前用户ID
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        // 调用 Service 层方法
        Page<Post> pageResult = postService.getRecommendedPosts(page, size, userId);
        return ResponseEntity.ok(pageResult);
    }
    @GetMapping("/recommended-by-tags")
    public ResponseEntity<Page<Post>> getRecommendedByTags(
            @RequestParam(defaultValue = "1") int page,
            @RequestParam(defaultValue = "10") int size) {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        long userId = (long) authentication.getPrincipal();
        // 调用标签推荐方法
        Page<Post> result = postService.getRecommendedByTags(page, size, userId);
        return ResponseEntity.ok(result);
    }
}