package com.example.g8backend.service;

import com.example.g8backend.entity.PostView;
import com.example.g8backend.mapper.*;
import com.example.g8backend.service.impl.PostServiceImpl;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentMatcher;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
@Transactional
public class PostHistoryServiceTest {

    @Mock
    private PostViewMapper postViewMapper;

    @Mock
    private PostMapper postMapper;

    @Mock
    private PostTagMapper postTagMapper;

    @Mock
    private UserTagPreferenceMapper userTagPreferenceMapper;

    @InjectMocks
    private PostServiceImpl postService;

    // ------------------------ 已有测试用例 ------------------------
    @Test
    public void testRecordViewHistory_NormalCase() {
        // 测试数据
        Long userId = 1L;
        Long postId = 100L;

        // 调用方法
        postService.recordViewHistory(userId, postId);

        // 验证行为
        verify(postViewMapper, times(1)).insert(any(PostView.class));
        verify(postMapper, times(1)).incrementViewCount(eq(postId));
    }

    @Test
    public void testRecordViewHistory_CheckDataIntegrity() {
        Long userId = 2L;
        Long postId = 200L;

        postService.recordViewHistory(userId, postId);

        // 显式指定参数类型为 PostView
        verify(postViewMapper).insert(argThat(new ArgumentMatcher<PostView>() {
            @Override
            public boolean matches(PostView view) {
                return view.getUserId().equals(userId) &&
                        view.getPostId().equals(postId) &&
                        view.getViewTime() != null;
            }
        }));
    }

    @Test
    public void testRecordViewHistory_MultipleCalls() {
        // 模拟多次调用
        Long postId = 300L;
        postService.recordViewHistory(1L, postId);
        postService.recordViewHistory(2L, postId);

        // 验证浏览数更新次数
        verify(postMapper, times(2)).incrementViewCount(postId);
    }

    // ------------------------ 新增测试用例 ------------------------
    @Test
    public void testRecordViewHistory_UpdateUserPreferenceWithTags() {
        // 测试数据
        Long userId = 3L;
        Long postId = 400L;
        List<Long> tagIds = Arrays.asList(10L, 20L);

        // 模拟标签查询
        when(postTagMapper.findTagIdsByPostId(postId)).thenReturn(tagIds);

        // 调用方法
        postService.recordViewHistory(userId, postId);

        // 验证标签偏好更新
        verify(userTagPreferenceMapper, times(2)).insertOrUpdateWeight(
                eq(userId),
                anyLong(),
                eq(0.1) // 假设每次浏览权重增加0.1
        );
    }

    @Test
    public void testRecordViewHistory_NoTagsShouldNotUpdatePreference() {
        // 测试数据
        Long userId = 4L;
        Long postId = 500L;

        // 模拟无关联标签
        when(postTagMapper.findTagIdsByPostId(postId)).thenReturn(Collections.emptyList());

        // 调用方法
        postService.recordViewHistory(userId, postId);

        // 验证未调用偏好更新
        verify(userTagPreferenceMapper, never()).insertOrUpdateWeight(
                anyLong(),
                anyLong(),
                anyDouble()
        );
    }

    @Test
    public void testRecordViewHistory_ConcurrentUpdateSafety() {
        // 测试数据
        Long postId = 600L;

        // 模拟多次并发调用
        postService.recordViewHistory(1L, postId);
        postService.recordViewHistory(2L, postId);
        postService.recordViewHistory(3L, postId);

        // 验证原子操作是否被正确调用
        verify(postMapper, times(3)).incrementViewCount(postId);
        verify(postViewMapper, times(3)).insert(any(PostView.class));
    }
}