package com.example.g8backend.service;
import com.example.g8backend.entity.Post;
import com.example.g8backend.mapper.*;
import com.example.g8backend.service.impl.PostServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.List;
import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;
@ExtendWith(SpringExtension.class)
@DisplayName("帖子服务测试")
class PostServiceTest {
    @Mock
    private PostMapper postMapper;
    @Mock
    private PostViewMapper postViewMapper;
    @Mock
    private CommentMapper commentMapper;
    private PostServiceImpl postService;
    private UserTagPreferenceMapper userTagPreferenceMapper;
    private PostTagMapper  postTagMapper;
    private Post testPost;
    @BeforeEach
    void setUp() {
        MockitoAnnotations.openMocks(this);
        postService = new PostServiceImpl(postMapper, postViewMapper,commentMapper,userTagPreferenceMapper,postTagMapper);
        testPost = createTestPost();
    }
    private Post createTestPost() {
        Post post = new Post();
        post.setPostId(1L);
        post.setUserId(1L);
        post.setPostTitle("测试标题");
        post.setPostContent("测试内容");
        post.setCreatedAt(new Timestamp(System.currentTimeMillis()));
        return post;
    }
    @Test
    @DisplayName("创建帖子-成功")
    void save_ShouldSucceed() {
        // Arrange
        when(postMapper.insert(any(Post.class))).thenReturn(1);
        // Act
        boolean result = postService.save(testPost);
        // Assert
        assertTrue(result);
        verify(postMapper).insert(testPost);
    }
    @Test
    @DisplayName("获取帖子-通过ID存在")
    void getById_WhenExists_ShouldReturnPost() {
        // Arrange
        when(postMapper.selectById(1L)).thenReturn(testPost);
        // Act
        Post result = postService.getById(1L);
        // Assert
        assertNotNull(result);
        assertEquals(testPost.getPostId(), result.getPostId());
        verify(postMapper).selectById(1L);
    }
    @Test
    @DisplayName("获取帖子-通过ID不存在")
    void getById_WhenNotExists_ShouldReturnNull() {
        // Arrange
        when(postMapper.selectById(999L)).thenReturn(null);
        // Act
        Post result = postService.getById(999L);
        // Assert
        assertNull(result);
        verify(postMapper).selectById(999L);
    }
    @Test
    @DisplayName("更新帖子-成功")
    void updateById_ShouldSucceed() {
        // Arrange
        when(postMapper.updateById(any(Post.class))).thenReturn(1);
        // Act
        boolean result = postService.updateById(testPost);
        // Assert
        assertTrue(result);
        verify(postMapper).updateById(testPost);
    }
    @Test
    @DisplayName("删除帖子-成功")
    void removeById_ShouldSucceed() {
        // Arrange
        when(postMapper.deleteById(1L)).thenReturn(1);
        // Act
        boolean result = postService.removeById(1L);
        // Assert
        assertTrue(result);
        verify(postMapper).deleteById(1L);
    }
    @Test
    @DisplayName("获取用户帖子列表")
    void getPostsByUserId_ShouldReturnPosts() {
        // Arrange
        List<Post> expectedPosts = Arrays.asList(testPost);
        when(postMapper.getPostsByUserId(1L)).thenReturn(expectedPosts);
        // Act
        List<Post> result = postService.getPostsByUserId(1L);
        // Assert
        assertNotNull(result);
        assertFalse(result.isEmpty());
        assertEquals(testPost.getPostId(), result.get(0).getPostId());
        verify(postMapper).getPostsByUserId(1L);
    }
    @Test
    @DisplayName("获取用户帖子-空列表")
    void getPostsByUserId_WhenNoPosts_ShouldReturnEmptyList() {
        // Arrange
        when(postMapper.getPostsByUserId(999L)).thenReturn(Arrays.asList());
        // Act
        List<Post> result = postService.getPostsByUserId(999L);
        // Assert
        assertNotNull(result);
        assertTrue(result.isEmpty());
        verify(postMapper).getPostsByUserId(999L);
    }
    // 新增测试方法：搜索帖子，支持多个标签
    @Test
    @DisplayName("搜索帖子-通过关键词和多个标签")
    void searchPosts_WithKeywordAndTags_ShouldReturnPosts() {
        // Arrange
        List<Long> tagIds = Arrays.asList(1L, 2L); // 假设存在标签ID 1和2
        String keyword = "测试内容";
        String author = "作者";
        List<Post> expectedPosts = Arrays.asList(testPost);
        // 模拟PostMapper的searchPosts方法
        when(postMapper.searchPosts(keyword, tagIds, author)).thenReturn(expectedPosts);
        // Act
        List<Post> result = postService.searchPosts(keyword, tagIds, author);
        // Assert
        assertNotNull(result);
        assertFalse(result.isEmpty());
        assertEquals(testPost.getPostId(), result.get(0).getPostId());
        verify(postMapper).searchPosts(keyword, tagIds, author);
    }
    @Test
    @DisplayName("搜索帖子-没有匹配的帖子")
    void searchPosts_WhenNoPosts_ShouldReturnEmptyList() {
        // Arrange
        List<Long> tagIds = Arrays.asList(3L, 4L); // 假设标签ID 3和4没有匹配的帖子
        String keyword = "不存在的内容";
        String author = "不存在的作者";
        List<Post> expectedPosts = Arrays.asList(); // 没有匹配的帖子
        // 模拟PostMapper的searchPosts方法
        when(postMapper.searchPosts(keyword, tagIds, author)).thenReturn(expectedPosts);
        // Act
        List<Post> result = postService.searchPosts(keyword, tagIds, author);
        // Assert
        assertNotNull(result);
        assertTrue(result.isEmpty());
        verify(postMapper).searchPosts(keyword, tagIds, author);
    }
}