package com.example.g8backend.service;

import com.example.g8backend.controller.AuthController;
import com.example.g8backend.util.JwtUtil;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.WebMvcTest;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.test.web.servlet.MockMvc;

import static org.mockito.Mockito.*;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.post;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;
import static org.springframework.security.test.web.servlet.request.SecurityMockMvcRequestPostProcessors.csrf;

@WebMvcTest(AuthController.class)
@AutoConfigureMockMvc(addFilters = false) // 关键：禁用Security过滤器
class AuthControllerTest {

    @Autowired
    private MockMvc mockMvc;

    // 为所有依赖的 Bean 添加 Mock
    @MockBean
    private com.example.g8backend.service.IUserService userService;
    @MockBean
    private com.example.g8backend.service.IUserStatsService userStatsService;
    @MockBean
    private com.example.g8backend.util.mailUtil mailUtil;
    @MockBean
    private org.springframework.security.crypto.password.PasswordEncoder passwordEncoder;
    @MockBean
    private JwtUtil jwtUtil;
    @MockBean
    private org.springframework.data.redis.core.RedisTemplate<String, Object> redisTemplate;

    @BeforeEach
    void setUp() {
        // 默认设置为已认证用户
        Authentication authentication = mock(Authentication.class);
        when(authentication.getPrincipal()).thenReturn(123L);
        SecurityContext securityContext = mock(SecurityContext.class);
        when(securityContext.getAuthentication()).thenReturn(authentication);
        SecurityContextHolder.setContext(securityContext);
    }

    @Test
    @DisplayName("刷新token-认证通过返回新token")
    void refreshToken_ShouldReturnNewToken() throws Exception {
        String newToken = "new.jwt.token";
        when(jwtUtil.generateToken(123L)).thenReturn(newToken);

        mockMvc.perform(post("/auth/refresh-token")
                        .header("Authorization", "Bearer old.jwt.token")
                        .with(csrf())
                        .accept("application/json")) // 加上这一行
                        .andExpect(status().isOk())
                .andExpect(jsonPath("$.code").value(200))
                .andExpect(jsonPath("$.message").value("Token刷新成功"))
                .andExpect(jsonPath("$.data").value(newToken));
    }

    @Test
    @DisplayName("刷新token-未认证返回401")
    void refreshToken_Unauthenticated_ShouldReturn401() throws Exception {
        // 设置为未认证
        Authentication authentication = mock(Authentication.class);
        when(authentication.getPrincipal()).thenReturn("anonymousUser");
        SecurityContext securityContext = mock(SecurityContext.class);
        when(securityContext.getAuthentication()).thenReturn(authentication);
        SecurityContextHolder.setContext(securityContext);

        mockMvc.perform(post("/auth/refresh-token")
                .with(csrf())
                .accept("application/json")) // 加上这一行
                .andExpect(status().isOk())
                .andExpect(jsonPath("$.code").value(401))
                .andExpect(jsonPath("$.message").value("未认证，无法刷新token"));
    }
}