package com.example.g8backend.service;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.example.g8backend.entity.User;
import com.example.g8backend.entity.UserSignin;
import com.example.g8backend.mapper.UserMapper;
import com.example.g8backend.mapper.UserSigninMapper;
import com.example.g8backend.service.impl.SigninServiceImpl;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class SigninServiceImplTest {
    @Mock
    private UserMapper userMapper;
    @Mock
    private UserSigninMapper userSigninMapper;
    @InjectMocks
    private SigninServiceImpl signinService;

    @Test
    public void testSignIn_Success() {
        // 模拟用户初始状态
        User user = new User();
        user.setUserId(1L);
        user.setSigninCount(0);
        user.setUserLevel("lv1");

        when(userSigninMapper.existsByUserIdAndDate(eq(1L), any())).thenReturn(false);
        when(userMapper.selectById(1L)).thenReturn(user);

        boolean result = signinService.signIn(1L);
        assertTrue(result);
        assertEquals(1, user.getSigninCount());
    }

    @Test
    public void testSignIn_AlreadySigned() {
        when(userSigninMapper.existsByUserIdAndDate(eq(1L), any())).thenReturn(true);
        boolean result = signinService.signIn(1L);
        assertFalse(result);
    }

    @Test
    public void testSignIn_UpgradeToLv2() {
        User user = new User();
        user.setUserId(1L);
        user.setSigninCount(2); // 初始签到2次
        user.setUserLevel("lv1");

        when(userSigninMapper.existsByUserIdAndDate(eq(1L), any())).thenReturn(false);
        when(userMapper.selectById(1L)).thenReturn(user);

        boolean result = signinService.signIn(1L);
        assertTrue(result);
        assertEquals(3, user.getSigninCount());
        assertEquals("lv2", user.getUserLevel());
    }

    @Test
    public void testSignIn_UpgradeToLv3() {
        User user = new User();
        user.setUserId(1L);
        user.setSigninCount(9); // 初始签到9次
        user.setUserLevel("lv2");

        when(userSigninMapper.existsByUserIdAndDate(eq(1L), any())).thenReturn(false);
        when(userMapper.selectById(1L)).thenReturn(user);

        boolean result = signinService.signIn(1L);
        assertTrue(result);
        assertEquals(10, user.getSigninCount());
        assertEquals("lv3", user.getUserLevel());
    }
    @Test
    public void testGetSigninsByDateRange_NormalCase() {
        // 模拟输入参数
        Long userId = 1L;
        LocalDate startDate = LocalDate.of(2023, 10, 1);
        LocalDate endDate = LocalDate.of(2023, 10, 31);

        // 构建模拟数据
        UserSignin signin1 = new UserSignin();
        signin1.setSigninId(1L);
        signin1.setUserId(userId);
        signin1.setSigninDate(LocalDate.of(2023, 10, 5));

        UserSignin signin2 = new UserSignin();
        signin2.setSigninId(2L);
        signin2.setUserId(userId);
        signin2.setSigninDate(LocalDate.of(2023, 10, 15));

        List<UserSignin> mockSignins = Arrays.asList(signin1, signin2);

        when(userSigninMapper.selectList(any(QueryWrapper.class))).thenReturn(mockSignins);

        // 执行方法
        List<UserSignin> result = signinService.getSigninsByDateRange(userId, startDate, endDate);

        // 断言结果
        assertEquals(2, result.size());
        assertEquals(LocalDate.of(2023, 10, 5), result.get(0).getSigninDate());
        assertEquals(LocalDate.of(2023, 10, 15), result.get(1).getSigninDate());
        verify(userSigninMapper).selectList(any(QueryWrapper.class));
    }

    @Test
    public void testGetSigninsByDateRange_NoRecords() {
        Long userId = 1L;
        LocalDate startDate = LocalDate.of(2023, 11, 1);
        LocalDate endDate = LocalDate.of(2023, 11, 30);

        when(userSigninMapper.selectList(any())).thenReturn(Collections.emptyList());

        List<UserSignin> result = signinService.getSigninsByDateRange(userId, startDate, endDate);

        assertTrue(result.isEmpty());
    }

    @Test
    public void testGetSigninsByDateRange_SingleDay() {
        Long userId = 1L;
        LocalDate date = LocalDate.of(2023, 10, 10);

        UserSignin signin = new UserSignin();
        signin.setSigninId(3L);
        signin.setUserId(userId);
        signin.setSigninDate(date);

        // 修正：返回包含模拟数据的列表
        when(userSigninMapper.selectList(any(QueryWrapper.class))).thenReturn(Collections.singletonList(signin));

        List<UserSignin> result = signinService.getSigninsByDateRange(userId, date, date);

        assertEquals(1, result.size());
        assertEquals(date, result.get(0).getSigninDate());
    }
    @Test
    public void testGetSigninsByDateRange_InvalidDateOrder() {
        Long userId = 1L;
        LocalDate startDate = LocalDate.of(2023, 10, 31);
        LocalDate endDate = LocalDate.of(2023, 10, 1);

        when(userSigninMapper.selectList(any())).thenReturn(Collections.emptyList());

        List<UserSignin> result = signinService.getSigninsByDateRange(userId, startDate, endDate);

        assertTrue(result.isEmpty());
    }

    public void testGetSigninsByDateRange_SortedOrder() {
        Long userId = 1L;
        LocalDate startDate = LocalDate.of(2023, 10, 1);
        LocalDate endDate = LocalDate.of(2023, 10, 31);

        // 模拟数据并按日期升序排列
        UserSignin signin1 = new UserSignin();
        signin1.setSigninId(1L);
        signin1.setSigninDate(LocalDate.of(2023, 10, 5));

        UserSignin signin2 = new UserSignin();
        signin2.setSigninId(2L);
        signin2.setSigninDate(LocalDate.of(2023, 10, 15));

        List<UserSignin> mockSignins = Arrays.asList(signin1, signin2); // 直接提供已排序的数据

        when(userSigninMapper.selectList(any(QueryWrapper.class))).thenReturn(mockSignins);

        List<UserSignin> result = signinService.getSigninsByDateRange(userId, startDate, endDate);

        // 验证顺序
        assertEquals(LocalDate.of(2023, 10, 5), result.get(0).getSigninDate());
        assertEquals(LocalDate.of(2023, 10, 15), result.get(1).getSigninDate());
    }
}