package com.example.g8backend.service;

import com.example.g8backend.dto.TorrentRecommendationDTO;
import com.example.g8backend.entity.Torrent;
import com.example.g8backend.mapper.PeerMapper;
import com.example.g8backend.mapper.TorrentMapper;
import com.example.g8backend.service.impl.TorrentRecommendationServiceImpl;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class TorrentRecommendationServiceImplTest {

    @Mock
    private PeerMapper peerMapper;

    @Mock
    private TorrentMapper torrentMapper;

    @InjectMocks
    private TorrentRecommendationServiceImpl service;

    @Test
    void testGetColdTorrentRecommendations_withValidData() {
        List<String> infoHashes = Arrays.asList("hash1", "hash2");
        when(peerMapper.selectAllInfoHashesWithPeers()).thenReturn(infoHashes);

        LocalDateTime now = LocalDateTime.now();

        Torrent torrent1 = new Torrent();
        torrent1.setTorrentId(1L);
        torrent1.setTorrentName("Torrent1");
        torrent1.setInfoHash("hash1");
        torrent1.setFileSize(100.0);
        torrent1.setUploadTime(now.minusHours(48));

        Torrent torrent2 = new Torrent();
        torrent2.setTorrentId(2L);
        torrent2.setTorrentName("Torrent2");
        torrent2.setInfoHash("hash2");
        torrent2.setFileSize(200.0);
        torrent2.setUploadTime(now.minusHours(72));

        when(torrentMapper.selectByInfoHashList(infoHashes))
                .thenReturn(Arrays.asList(torrent1, torrent2));
        when(peerMapper.countRecentActivity("hash1")).thenReturn(4L);
        when(peerMapper.countRecentActivity("hash2")).thenReturn(2L);

        List<TorrentRecommendationDTO> recommendations = service.getColdTorrentRecommendations(2);

        assertNotNull(recommendations);
        assertEquals(2, recommendations.size());
        // 根据冷度得分，torrent2 排第一
        assertEquals("Torrent2", recommendations.get(0).getTorrentName());
        assertEquals("Torrent1", recommendations.get(1).getTorrentName());

        assertEquals(72.0 / 3, recommendations.get(0).getColdnessScore(), 0.01);
        assertEquals(48.0 / 5, recommendations.get(1).getColdnessScore(), 0.01);

        verify(peerMapper).selectAllInfoHashesWithPeers();
        verify(torrentMapper).selectByInfoHashList(infoHashes);
        verify(peerMapper).countRecentActivity("hash1");
        verify(peerMapper).countRecentActivity("hash2");
    }

    @Test
    void testGetColdTorrentRecommendations_noPeers() {
        when(peerMapper.selectAllInfoHashesWithPeers()).thenReturn(Collections.emptyList());

        List<TorrentRecommendationDTO> recommendations = service.getColdTorrentRecommendations(5);

        assertNotNull(recommendations);
        assertTrue(recommendations.isEmpty());

        verify(peerMapper).selectAllInfoHashesWithPeers();
        verifyNoInteractions(torrentMapper);
    }

    @Test
    void testGetColdTorrentRecommendations_uploadTimeNull_skips() {
        when(peerMapper.selectAllInfoHashesWithPeers()).thenReturn(Collections.singletonList("hash1"));

        Torrent torrent = new Torrent();
        torrent.setTorrentId(1L);
        torrent.setTorrentName("Torrent1");
        torrent.setInfoHash("hash1");
        torrent.setUploadTime(null);

        when(torrentMapper.selectByInfoHashList(anyList()))
                .thenReturn(Collections.singletonList(torrent));

        List<TorrentRecommendationDTO> recommendations = service.getColdTorrentRecommendations(10);

        assertNotNull(recommendations);
        assertTrue(recommendations.isEmpty());

        verify(peerMapper).selectAllInfoHashesWithPeers();
        verify(torrentMapper).selectByInfoHashList(anyList());
        verify(peerMapper, never()).countRecentActivity(anyString());
    }

}
