package com.example.g8backend.controller;

import com.example.g8backend.dto.ApiResponse;
import com.example.g8backend.entity.Post;
import com.example.g8backend.entity.Report;
import com.example.g8backend.service.AdminService;
import com.example.g8backend.service.IPostService;
import com.example.g8backend.service.IReportService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/admin")
public class AdminController {
    @Autowired
    private AdminService adminService;
    private IReportService reportService;
    @Autowired
    private IPostService postService;
    @PostMapping("/grant-vip/{userId}")
    @PreAuthorize("hasRole('ADMIN')") // 仅允许管理员访问
    public String grantVip(@PathVariable Long userId) {
        boolean success = adminService.grantVip(userId);
        return success ? "VIP授予成功" : "操作失败（用户不存在）";
    }
    // 获取举报记录（支持按状态过滤）
    @GetMapping("/reports")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<List<Report>> getReports(
            @RequestParam(required = false) String status) {
        Long adminId = (Long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        return ApiResponse.success(reportService.getReports(status, adminId));
    }
    // 处理举报
    @PutMapping("/reports/{reportId}")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<String> resolveReport(
            @PathVariable Long reportId,
            @RequestParam String status,
            @RequestParam(required = false) String notes) {
        Long adminId = (Long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();//这里之前不太对，service改了
        reportService.resolveReport(reportId, adminId, status, notes);
        return ApiResponse.success("举报处理完成");
    }


    // 封禁用户
    @PostMapping("/users/{userId}/ban")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<String> banUser(
            @PathVariable Long userId,
            @RequestParam String reason) {
        Long adminId = getCurrentAdminId();
        boolean success = adminService.banUser(userId, reason, adminId);
        return success ?
                ApiResponse.success("用户封禁成功") :
                ApiResponse.error(400, "操作失败");
    }

    // 解封用户
    @PostMapping("/users/{userId}/unban")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<String> unbanUser(@PathVariable Long userId) {
        Long adminId = getCurrentAdminId();
        boolean success = adminService.unbanUser(userId, adminId);
        return success ?
                ApiResponse.success("用户解封成功") :
                ApiResponse.error(400, "操作失败");
    }

    // 锁定帖子
    @PostMapping("/posts/{postId}/lock")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<String> lockPost(
            @PathVariable Long postId,
            @RequestParam String reason) {
        Long adminId = getCurrentAdminId();
        boolean success = adminService.lockPost(postId, reason, adminId);
        return success ?
                ApiResponse.success("帖子已锁定") :
                ApiResponse.error(400, "操作失败");
    }

    // 解锁帖子
    @PostMapping("/posts/{postId}/unlock")
    @PreAuthorize("hasRole('ADMIN')")
    public ApiResponse<String> unlockPost(@PathVariable Long postId) {
        Long adminId = getCurrentAdminId();
        boolean success = adminService.unlockPost(postId, adminId);
        return success ?
                ApiResponse.success("帖子已解锁") :
                ApiResponse.error(400, "操作失败");
    }
    @DeleteMapping("/{postId}")
    public ResponseEntity<ApiResponse<String>> deletePost(@PathVariable Long postId) {
        long userId = (long) SecurityContextHolder.getContext().getAuthentication().getPrincipal();
        Post post = postService.getById(postId);
        if (post == null) {
            return ResponseEntity.status(404).body(ApiResponse.error(404, "Post not found."));
        }
        postService.removeById(postId);
        return ResponseEntity.ok(ApiResponse.message("Post deleted successfully."));
    }

    private Long getCurrentAdminId() {
        return (Long) SecurityContextHolder.getContext()
                .getAuthentication().getPrincipal();
    }

}