package com.example.g8backend.service;

import com.example.g8backend.dto.AnnounceRequestDTO;
import com.example.g8backend.dto.AnnounceResponseDTO;
import com.example.g8backend.entity.Peer;
import com.example.g8backend.mapper.PeerMapper;
import com.example.g8backend.service.impl.TrackerServiceImpl;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.*;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.SetOperations;

import java.util.*;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
public class TrackerServiceTest {

    @InjectMocks
    private TrackerServiceImpl trackerService;

    @Mock
    private PeerMapper peerMapper;

    @Mock
    private RedisTemplate<String, Object> redisTemplate;

    @Mock
    private SetOperations<String, Object> setOperations;

    @BeforeEach
    public void setUp() {
        when(redisTemplate.opsForSet()).thenReturn(setOperations);
    }

    @Test
    public void testHandleAnnounceStoppedEvent() {
        AnnounceRequestDTO requestDTO = new AnnounceRequestDTO();
        requestDTO.setEvent("stopped");
        requestDTO.setInfoHash("infohash");
        requestDTO.setPeerId("peer1");

        AnnounceResponseDTO response = trackerService.handleAnnounce(requestDTO);

        verify(setOperations).remove("peers:infohash", "peer1");
        assertEquals(30, response.getInterval());
        assertTrue(response.getPeers().isEmpty());
    }

    @Test
    public void testHandleAnnounceWithNewPeer() {
        AnnounceRequestDTO requestDTO = new AnnounceRequestDTO();
        requestDTO.setPasskey("key123");
        requestDTO.setInfoHash("infohash");
        requestDTO.setPeerId("peer1");
        requestDTO.setIp("192.168.0.1");
        requestDTO.setPort(6881);
        requestDTO.setUploaded(1000);
        requestDTO.setDownloaded(500);
        requestDTO.setEvent("started");

        when(peerMapper.getPeerByPK("peer1", "infohash", "key123")).thenReturn(null);
        when(setOperations.members("peers:infohash")).thenReturn(Set.of("peer1"));
        when(peerMapper.getPeerByInfoHashAndPeerId("infohash", "peer1")).thenReturn(List.of(new Peer() {{
            setIpAddress("192.168.0.1");
            setPort(6881);
        }}));

        AnnounceResponseDTO response = trackerService.handleAnnounce(requestDTO);

        verify(peerMapper).insert(any(Peer.class));
        verify(setOperations).add("peers:infohash", "peer1");
        assertEquals(30, response.getInterval());
        assertEquals(1, response.getPeers().size());
        assertEquals("192.168.0.1", response.getPeers().get(0).get("ip"));
    }

    @Test
    public void testHandleAnnounceWithExistingPeer() {
        AnnounceRequestDTO requestDTO = new AnnounceRequestDTO();
        requestDTO.setPasskey("key123");
        requestDTO.setInfoHash("infohash");
        requestDTO.setPeerId("peer2");
        requestDTO.setIp("192.168.1.1");
        requestDTO.setPort(6882);
        requestDTO.setUploaded(2000);
        requestDTO.setDownloaded(1000);
        requestDTO.setEvent("update");

        Peer existing = new Peer();
        existing.setPeerId("peer2");
        existing.setPasskey("key123");
        existing.setInfo_hash("infohash");
        existing.setUploaded(1000.0);
        existing.setDownloaded(500.0);

        when(peerMapper.getPeerByPK("peer2", "infohash", "key123")).thenReturn(existing);
        when(setOperations.members("peers:infohash")).thenReturn(Set.of("peer2"));
        when(peerMapper.getPeerByInfoHashAndPeerId("infohash", "peer2")).thenReturn(List.of(new Peer() {{
            setIpAddress("192.168.1.1");
            setPort(6882);
        }}));

        AnnounceResponseDTO response = trackerService.handleAnnounce(requestDTO);

        verify(peerMapper).updatePeer(eq("key123"), eq("peer2"), eq("infohash"), eq(2000.0), eq(1000.0));
        verify(setOperations).add("peers:infohash", "peer2");
        assertEquals(1, response.getPeers().size());
        assertEquals("192.168.1.1", response.getPeers().get(0).get("ip"));
    }

    @Test
    public void testHandleAnnounceExceptionInRedis() {
        AnnounceRequestDTO requestDTO = new AnnounceRequestDTO();
        requestDTO.setPasskey("keyX");
        requestDTO.setInfoHash("hashX");
        requestDTO.setPeerId("peerX");
        requestDTO.setIp("10.0.0.1");
        requestDTO.setPort(6883);
        requestDTO.setUploaded(100);
        requestDTO.setDownloaded(200);
        requestDTO.setEvent("started");

        when(peerMapper.getPeerByPK(any(), any(), any())).thenReturn(null);
        when(setOperations.members(anyString())).thenReturn(new HashSet<>() );

        AnnounceResponseDTO response = trackerService.handleAnnounce(requestDTO);

        assertEquals(30, response.getInterval());
        assertTrue(response.getPeers().isEmpty());
    }
}
