import instance from '@/utils/axios';
import React, { useCallback, useEffect, useState } from 'react';
import styles from './upload.module.css';
import {postTorrentUpload } from '@/api/torrent';
import { useLocation, useNavigate } from 'react-router-dom'; // 用于跳转
import { getTagsByMainTag } from '@/utils/common';
import { Checkbox, Row, Col } from 'antd'; // 使用 antd 的 Checkbox 组件
import { MainTag } from '@/types/common';
import { Select } from 'antd';
import { UploadOutlined } from '@ant-design/icons';
import type { UploadProps } from 'antd';
import { Button, message, Upload as UploadArea } from 'antd';


const CreatePost = () => {
  const location = useLocation();
  const [postTitle, setPostTitle] = useState('');
  const [postContent, setPostContent] = useState('');
  const [isChecked, setIsChecked] = useState(false);
  const {type} = location.state || {type: ''}; // 获取路由传递的参数
  const [tagIds, setTagIds] = useState<number[]>([]);
  const [postType, setPostType] = useState<string>(type || ''); // 初始化 postType
  const navigate = useNavigate();
  const [tags, setTags] = useState<Map<string, number>>(new Map());

  const props: UploadProps = {
    name: 'file',
    action: process.env.REACT_APP_BASE_URL + postTorrentUpload, // 替换为实际的上传接口地址
    headers: {
      authorization: 'authorization-text',
    },
    onChange(info) {
      if (info.file.status !== 'uploading') {
        console.log(info.file, info.fileList);
      }
      if (info.file.status === 'done') {
        message.success(`${info.file.name} file uploaded successfully`);
      } else if (info.file.status === 'error') {
        message.error(`${info.file.name} file upload failed.`);
      }
    },
  };


  const handleSubmit = async () => {
    if (!postTitle.trim() || !postType || !postContent.trim()) {
      alert('请填写完整内容（资源名、类型、内容介绍）');
      return;
    }

    if (!isChecked) {
      alert('请先确认您已知晓以上内容');
      return;
    }

    const payload = {
      post: {
        postId: 0,
        userId: 0,
        postTitle,
        postContent,
        createdAt: Date.now(),
        postType,
        viewCount: 0,
        hotScore: 5,
        lastCalculated: Date.now()
      },
      tagIds: [0]
    };

    try {
      const res = await instance.post(Upload, payload);

      console.log('mock返回内容：', res.code);

      // 判断返回内容是否成功（根据你 mock 接口返回的 code 字段）
      if (res.code !== 0) throw new Error('发布失败');

      alert('发布成功！');
      navigate(-1); // 返回上一页（homepage）
    } catch (error) {
      alert('发布失败，请稍后重试');
      console.error(error);
    }
  };

  useEffect(() => {
    setTags(getTagsByMainTag(type)); // 获取对应类型的标签
  },[]);

  const handlePostTypeChange = useCallback((value:string) => {
    setPostType(value);
    setTags(getTagsByMainTag(value)); // 更新 tags
    setTagIds([]); // 清空已选的标签 ID
  }, [postType, setTags, setTagIds]);


  return (
    <div className={styles.container}>
      <div className={styles.formGroup}>
        <label>资源名：</label>
        <input
          type="text"
          value={postTitle}
          placeholder="请输入文本"
          onChange={(e) => setPostTitle(e.target.value)}
          className={styles.input}
        />
      </div>
  
      <div className={styles.formGroup}>
        <label>种子类型：</label>
        <Select
          value={postType}
          onChange={handlePostTypeChange}

          options ={ MainTag.map((item) => ({
            label: item,
            value: item
          }))} />
      </div>

      <div className={styles.formGroup}>
        <label>类型选择：</label>
        <Checkbox.Group
          value={tagIds}
          onChange={(checkedValues) => setTagIds(checkedValues as number[])}
        >
          <Row gutter={[12, 12]}>
            {[...tags.entries()].map(([name, id]) => (
              <Col key={id} xs={12} sm={8} md={6} lg={4}>
                <Checkbox value={id}>{name}</Checkbox>
              </Col>
            ))}
          </Row>
        </Checkbox.Group>
      </div>

      {/* 暂时移除上传文件表单 */}
      <div className={styles.formGroup}>
        <label>上传资源：</label>
        <UploadArea {...props}>
          <Button icon={<UploadOutlined />}>Upload</Button>
        </UploadArea>
      </div>

      <div className={styles.formGroup}>
        <label>内容介绍：</label>
        <textarea
          placeholder="请输入内容介绍"
          value={postContent}
          onChange={(e) => setPostContent(e.target.value)}
          maxLength={200}
          className={styles.textarea}
        />
        <div className={styles.charCount}>{postContent.length}/200</div>
      </div>

      <div className={styles.requirement}>【发布内容要求】</div>

      <div className={styles.checkbox}>
        <input
          type="checkbox"
          checked={isChecked}
          onChange={() => setIsChecked(!isChecked)}
        />
        <span>我已知晓以上内容</span>
      </div>

      <button onClick={handleSubmit} className={styles.submitBtn} disabled={!isChecked}>
        我已知晓
      </button>
    </div>
  );
};

export default CreatePost;
