import React, { useEffect, useRef, useState } from 'react';
import styles from './pet.module.css'; // 确保这个路径对

const PetGame: React.FC = () => {
  const petRef = useRef<HTMLDivElement>(null);
  const [mood, setMood] = useState(50);
  const [hunger, setHunger] = useState(50);

  // 从 localStorage 恢复状态
  useEffect(() => {
    const saved = localStorage.getItem('petState');
    if (saved) {
      const { mood: savedMood, hunger: savedHunger } = JSON.parse(saved);
      setMood(savedMood);
      setHunger(savedHunger);
    }
  }, []);

  // 每分钟更新状态并保存
  useEffect(() => {
    const interval = setInterval(() => {
      setMood(prev => {
        const newMood = Math.min(prev + 1, 100);
        savePetState(newMood, hunger);
        return newMood;
      });
      setHunger(prev => {
        const newHunger = Math.max(prev - 1, 0);
        savePetState(mood, newHunger);
        return newHunger;
      });
    }, 60000); // 每分钟一次

    return () => clearInterval(interval);
  }, [mood, hunger]);

  // 保存状态到 localStorage
  const savePetState = (newMood: number, newHunger: number) => {
    localStorage.setItem('petState', JSON.stringify({ mood: newMood, hunger: newHunger }));
  };

  // 拖动功能（保留）
  useEffect(() => {
    const pet = petRef.current;
    if (!pet) return;

    let offsetX = 0;
    let offsetY = 0;
    let isDragging = false;

    const handleMouseDown = (e: MouseEvent) => {
      isDragging = true;
      const rect = pet.getBoundingClientRect();
      offsetX = e.clientX - rect.left;
      offsetY = e.clientY - rect.top;
    };

    const handleMouseMove = (e: MouseEvent) => {
      if (isDragging && pet) {
        pet.style.left = `${e.clientX - offsetX}px`;
        pet.style.top = `${e.clientY - offsetY}px`;
      }
    };

    const handleMouseUp = () => {
      isDragging = false;
    };

    pet.addEventListener('mousedown', handleMouseDown);
    document.addEventListener('mousemove', handleMouseMove);
    document.addEventListener('mouseup', handleMouseUp);

    return () => {
      pet.removeEventListener('mousedown', handleMouseDown);
      document.removeEventListener('mousemove', handleMouseMove);
      document.removeEventListener('mouseup', handleMouseUp);
    };
  }, []);

  // 喂食按钮逻辑
  const handleFeed = () => {
    const newHunger = Math.min(hunger + 10, 100);
    setHunger(newHunger);
    savePetState(mood, newHunger);
  };

  // 互动按钮逻辑
  const handleInteract = () => {
    const newMood = Math.min(mood + 10, 100);
    setMood(newMood);
    savePetState(newMood, hunger);
  };

  return (
    <div className={styles.cardContainer}>
      <div className={styles.card}>
        <p className={styles.text}>情绪：{mood}</p>
        <p className={styles.text}>饥饿值：{hunger}/100</p>

        <button className={styles.feedButton} onClick={handleFeed}>
          🍖 喂食
        </button>

        <button className={styles.feedButton} onClick={handleInteract}>
          🐾 摸摸头 / 玩游戏
        </button>
      </div>
    </div>
  );
};

export default PetGame;
