import { useState, useEffect, useCallback } from 'react'
import { data } from 'react-router'

// type RequestFunction<T> = () => Promise<T>
type RequestFunction<T, P = any> = (params?: P) => Promise<T>;

interface UseApiResult<T> {
  data: T | null
  loading: boolean
  error: Error | null
  refresh: () => void
}

// export function useApi<T>(
//     requestFn: RequestFunction<T>,
//     immediate = true
//   ): UseApiResult<T> {
export function useApi<T, P = any>(
  requestFn: RequestFunction<T, P>,  // 接收参数
  immediate = true
) {
    const [data, setData] = useState<T | null>(null)
    const [loading, setLoading] = useState(false)
    const [error, setError] = useState<Error | null>(null)
  
    const execute = useCallback(async (params?: P) => { // 添加参数
      try {
        setLoading(true)
        const result = await requestFn(params);//传参
        setData(result)
        setError(null)
        return result  // 返回请求结果
      } catch (err) {
        setError(err as Error)
        throw err
      } finally {
        setLoading(false)
      }
    }, [requestFn])
  
    useEffect(() => {
      if (immediate) {
        execute()
      }
    }, [execute, immediate])
  
    return { data, loading, error, refresh: execute }
  }