import { getRefreshToken } from '@/api/auth';
import axios from 'axios';

function parseJwt(token: string) {
    console.log('Parsing JWT token:', token);
    try {
        const base64Url = token.split('.')[1];
        const base64 = base64Url.replace(/-/g, '+').replace(/_/g, '/');
        const jsonPayload = decodeURIComponent(
            atob(base64)
                .split('')
                .map(c => `%${('00' + c.charCodeAt(0).toString(16)).slice(-2)}`)
                .join('')
        );
        return JSON.parse(jsonPayload);
    } catch (error) {
        console.error('Invalid JWT token:', error);
        return null;
    }
}

/**
 * 检查token是否过期
 * @returns {string} -1: token无效 0: token未过期 1: token已过期
**/
export function isTokenExpired(token: string): string {
    if (token === null || token === undefined) {
        return '-1'; // 如果没有token，认为token无效
    }
    const decoded = parseJwt(token);
    if (!decoded || !decoded.exp) {
        return '-1'; // 如果解析失败或没有 `exp` 字段，认为token无效
    }
    const currentTime = Math.floor(Date.now() / 1000); // 当前时间（秒）
    const bufferTime = 60 * 10; // 过期前的缓冲时间（秒）
    return decoded.exp < currentTime ?decoded.exp < currentTime - bufferTime ? '-1' : '1'  :"0";
}

async function refreshToken() {
    try {
        const response = await axios.post(getRefreshToken, {
            token: localStorage.getItem('token'),
        });
        const newToken = response.data.token;
        localStorage.setItem('token', newToken);
        return newToken;
    } catch (error) {
        console.error('Failed to refresh token:', error);
        return null;
    }
}

export async function checkAndRefreshToken() {
    const token = localStorage.getItem('token');
    if (token && isTokenExpired(token)=='1') {
        const newToken = await refreshToken();
        if (!newToken) {
            localStorage.removeItem('token');
        }
    }
}