package com.g9.g9backend.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.g9.g9backend.pojo.Community;
import com.g9.g9backend.pojo.DTO.*;
import com.g9.g9backend.pojo.Subscription;
import com.g9.g9backend.pojo.Thread;
import com.g9.g9backend.pojo.ThreadLike;
import com.g9.g9backend.service.*;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;

/**
 * CommunityController 社区控制器类，处理与社区相关的请求
 *
 * @author Seamher
 */
@RestController
public class CommunityController {

    private final CommunityService communityService;

    private final ThreadService threadService;

    private final SubscriptionService subscriptionService;

    private final ThreadLikeService threadLikeService;

    private final Logger logger = LoggerFactory.getLogger(CommunityController.class);

    public CommunityController(CommunityService communityService, ThreadService threadService, SubscriptionService subscriptionService, ThreadLikeService threadLikeService) {
        this.communityService = communityService;
        this.threadService = threadService;
        this.threadLikeService = threadLikeService;
        this.subscriptionService = subscriptionService;
    }

    @PostMapping(value = "/thread")
    public ResponseEntity<String> postThread(@RequestBody Thread thread) {
        thread.setLikes(0);
        thread.setCommentNumber(0);
        threadService.save(thread);

        Community community = communityService.getById(thread.getCommunityId());
        community.setThreadNumber(community.getThreadNumber() + 1);
        communityService.updateById(community);

        logger.info("帖子已发布");

        return ResponseEntity.ok("");
    }

    @PostMapping(value = "/thread/like")
    public ResponseEntity<String> postThreadLike(@RequestBody Map<String, Integer> request) {
        Integer userId = request.get("userId");
        Integer threadId = request.get("threadId");

        ThreadLike threadLike = new ThreadLike(userId, threadId);
        threadLikeService.save(threadLike);

        Thread thread = threadService.getById(threadId);
        thread.setLikes(thread.getLikes() + 1);
        threadService.updateById(thread);

        logger.info("点赞成功");

        return ResponseEntity.ok("");
    }

    @DeleteMapping(value = "/thread")
    public ResponseEntity<String> deleteThread(@RequestParam Integer threadId) {
        Thread thread = threadService.getById(threadId);
        threadService.removeById(threadId);

        Community community = communityService.getById(thread.getCommunityId());
        community.setThreadNumber(community.getThreadNumber() - 1);
        communityService.updateById(community);

        logger.info("帖子已删除");

        return ResponseEntity.noContent().build();
    }

    @DeleteMapping(value = "/thread/like")
    public ResponseEntity<String> deleteLike(@RequestParam Integer userId, @RequestParam Integer threadId) {
        LambdaQueryWrapper<ThreadLike> threadLikeQuery = new LambdaQueryWrapper<ThreadLike>()
                .eq(ThreadLike::getUserId, userId)
                .eq(ThreadLike::getThreadId, threadId);

        threadLikeService.remove(threadLikeQuery);

        Thread thread = threadService.getById(threadId);
        thread.setLikes(thread.getLikes() - 1);
        threadService.updateById(thread);

        logger.info("取消点赞成功");

        return ResponseEntity.noContent().build();
    }

    @GetMapping(value = "/community")
    public ResponseEntity<GetCommunityDTO> getCommunity(@RequestParam String searchValue,
                                                        @RequestParam String type,
                                                        @RequestParam Integer pageNumber,
                                                        @RequestParam Integer rows) {
        Page<Community> communityPage = new Page<>(pageNumber, rows);
        LambdaQueryWrapper<Community> communityQuery = new LambdaQueryWrapper<Community>()
                .eq(Community::getType, type)
                .like(StringUtils.isNotBlank(searchValue), Community::getCommunityName, searchValue)
                .orderByDesc(Community::getHot);

        Page<Community> result = communityService.page(communityPage, communityQuery);

        GetCommunityDTO getCommunityDTO = wrapCommunityPage(result, item -> {
            GetCommunityDTO.Community community = new GetCommunityDTO.Community();
            community.setCommunityId(item.getCommunityId());
            community.setCommunityName(item.getCommunityName());
            community.setCommunityPicture(item.getCommunityPicture());
            community.setDescription(item.getDescription());
            community.setHot(item.getHot());
            community.setThreadNumber(item.getThreadNumber());
            community.setResourceId(item.getResourceId());
            return community;
        });

        return ResponseEntity.ok(getCommunityDTO);
    }

    @NotNull
    private <T> GetCommunityDTO wrapCommunityPage(Page<T> page, Function<T, GetCommunityDTO.Community> mapper) {
        List<GetCommunityDTO.Community> records = page.getRecords().stream().map(mapper).toList();

        return new GetCommunityDTO(records, (int) page.getTotal(), (int) page.getPages(), (int) page.getCurrent(), (int) page.getSize());
    }

    @GetMapping(value = "/community/info")
    public ResponseEntity<Community> getCommunityInfo(@RequestParam Integer communityId) {

        return ResponseEntity.ok(communityService.getById(communityId));
    }

    @GetMapping(value = "/thread")
    public ResponseEntity<ThreadDTO> getThread(@RequestParam Integer threadId, @RequestParam Integer userId) {
        LambdaQueryWrapper<ThreadLike> threadLikeQuery = new LambdaQueryWrapper<ThreadLike>()
                .eq(ThreadLike::getUserId, userId)
                .eq(ThreadLike::getThreadId, threadId);
        Thread thread = threadService.getById(threadId);
        ThreadDTO threadDTO = new ThreadDTO(
                threadId,
                thread.getUserId(),
                thread.getThreadPicture(),
                thread.getTitle(),
                thread.getContent(),
                thread.getLikes(),
                threadLikeService.getOne(threadLikeQuery) != null,
                thread.getCreateAt(),
                thread.getCommentNumber(),
                thread.getCommunityId()
        );

        return ResponseEntity.ok(threadDTO);
    }

    @GetMapping(value = "/community/threads")
    public ResponseEntity<GetCommunityThreadsDTO> getCommunityThreads(@RequestParam Integer communityId,
                                                                      @RequestParam Integer pageNumber,
                                                                      @RequestParam Integer rows,
                                                                      @RequestParam String option,
                                                                      @RequestParam String searchValue,
                                                                      @RequestParam Integer userId) {
        Page<Thread> threadPage = new Page<>(pageNumber, rows);
        LambdaQueryWrapper<Thread> threadQuery = new LambdaQueryWrapper<Thread>()
                .eq(Thread::getCommunityId, communityId)
                .like(StringUtils.isNotBlank(searchValue), Thread::getTitle, searchValue);

        if (Objects.equals(option, "最高热度")) {

            threadQuery.orderByDesc(Thread::getLikes);
        } else {

            List<Integer> userIds = subscriptionService.list(new LambdaQueryWrapper<Subscription>()
                            .eq(Subscription::getUserId, userId))
                            .stream()
                            .map(Subscription::getFollowerId)
                            .toList();

            threadQuery.in(!userIds.isEmpty(), Thread::getUserId, userIds)
                    .orderByDesc(Thread::getLikes);
        }

        Page<Thread> result = threadService.page(threadPage, threadQuery);

        GetCommunityThreadsDTO getCommunityThreadsDTO = wrapThreadPage(result, item -> {
            GetCommunityThreadsDTO.Thread thread = new GetCommunityThreadsDTO.Thread();
            thread.setThreadId(item.getThreadId());
            thread.setUserId(item.getUserId());
            thread.setThreadPicture(item.getThreadPicture());
            thread.setTitle(item.getTitle());
            thread.setLikes(item.getLikes());
            thread.setCreateAt(item.getCreateAt());
            return thread;
        });

        return ResponseEntity.ok(getCommunityThreadsDTO);
    }

    @NotNull
    private <T> GetCommunityThreadsDTO wrapThreadPage(Page<T> page, Function<T, GetCommunityThreadsDTO.Thread> mapper) {
        List<GetCommunityThreadsDTO.Thread> records = page.getRecords().stream().map(mapper).toList();

        return new GetCommunityThreadsDTO(records, (int) page.getTotal(), (int) page.getPages(), (int) page.getCurrent(), (int) page.getSize());
    }

    @GetMapping(value = "/community/hot")
    public ResponseEntity<GetCommunityHotDTO> getCommunityHot() {
        Page<Community> communityPage = new Page<>(1, 3);
        LambdaQueryWrapper<Community> communityQuery = new LambdaQueryWrapper<Community>()
                .orderByDesc(Community::getHot);

        Page<Community> result = communityService.page(communityPage, communityQuery);

        List<GetCommunityHotDTO.Community> communityList = getCommunityList(result);

        GetCommunityHotDTO getCommunityHotDTO = new GetCommunityHotDTO(communityList);

        return ResponseEntity.ok(getCommunityHotDTO);
    }

    @NotNull
    private static List<GetCommunityHotDTO.Community> getCommunityList(Page<Community> communityPage) {
        List<Community> communityListT = communityPage.getRecords();
        List<GetCommunityHotDTO.Community> communityList = new ArrayList<>();
        int status = 1;
        for (Community community : communityListT) {
            GetCommunityHotDTO.Community communityHot = new GetCommunityHotDTO.Community(
                    community.getCommunityId(),
                    community.getCommunityName(),
                    community.getCommunityPicture(),
                    community.getCommunityPicture(),
                    community.getHot(),
                    community.getType(),
                    community.getThreadNumber(),
                    community.getResourceId(),
                    status);
            communityList.add(communityHot);
            status++;
        }

        return communityList;
    }

    @GetMapping(value = "/community/common")
    public ResponseEntity<GetCommunityCommonDTO> getCommunityCommon() {
        Page<Community> communityPage = new Page<>(2, 3);
        LambdaQueryWrapper<Community> communityQuery = new LambdaQueryWrapper<Community>()
                .orderByDesc(Community::getHot);

        Page<Community> result = communityService.page(communityPage, communityQuery);

        List<Community> communityList = result.getRecords();

        GetCommunityCommonDTO getCommunityCommonDTO = new GetCommunityCommonDTO(communityList);

        return ResponseEntity.ok(getCommunityCommonDTO);
    }

}
