package com.g9.g9backend.controller;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.StringUtils;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.g9.g9backend.pojo.DTO.GetRewardDTO;
import com.g9.g9backend.pojo.Reward;
import com.g9.g9backend.service.RewardService;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

/**
 * RewardController 悬赏控制器类，处理与悬赏相关的请求
 *
 * @author Seamher
 */
@RestController
@RequestMapping("/reward")
public class RewardController {

    private final RewardService rewardService;

    private final Logger logger = LoggerFactory.getLogger(RewardController.class);

    public RewardController(RewardService rewardService) {
        this.rewardService = rewardService;
    }

    @PostMapping
    public ResponseEntity<String> postReward(@RequestBody Reward reward) {
        reward.setCompletedBy(null);
        reward.setResourceId(null);
        rewardService.save(reward);

        logger.info("悬赏已发布");

        return ResponseEntity.ok("");
    }

    @DeleteMapping
    public ResponseEntity<String> deleteReward(@RequestParam Integer rewardId) {
        rewardService.removeById(rewardId);

        logger.info("悬赏已删除");

        return ResponseEntity.noContent().build();
    }

    @GetMapping
    public ResponseEntity<GetRewardDTO> getReward(@RequestParam Integer pageNumber,
                                                  @RequestParam Integer rows,
                                                  @RequestParam String searchValue,
                                                  @RequestParam String option) {
        Page<Reward> rewardPage = new Page<>(pageNumber, rows);
        LambdaQueryWrapper<Reward> rewardQuery = new LambdaQueryWrapper<Reward>()
                .like(StringUtils.isNotBlank(searchValue), Reward::getRewardName, searchValue);

        if (Objects.equals(option, "赏金最高")) {

            rewardQuery.orderByDesc(Reward::getPrice);
        } else {

            rewardQuery.orderByDesc(Reward::getCreateAt);
        }

        Page<Reward> result = rewardService.page(rewardPage, rewardQuery);

        GetRewardDTO getRewardDTO = wrapRewardPage(result, item -> {
            GetRewardDTO.Reward reward = new GetRewardDTO.Reward();
            reward.setRewardId(item.getRewardId());
            reward.setRewardName(item.getRewardName());
            reward.setRewardPicture(item.getRewardPicture());
            reward.setUserId(item.getUserId());
            reward.setPrice(item.getPrice());
            reward.setCreateAt(item.getCreateAt());
            return reward;
        });

        return ResponseEntity.ok(getRewardDTO);
    }

    @NotNull
    private <T> GetRewardDTO wrapRewardPage(Page<T> page, Function<T, GetRewardDTO.Reward> mapper) {
        List<GetRewardDTO.Reward> records = page.getRecords().stream().map(mapper).toList();

        return new GetRewardDTO(records, (int) page.getTotal(), (int) page.getPages(), (int) page.getCurrent(), (int) page.getSize());
    }

    @GetMapping(value = "/info")
    public ResponseEntity<Reward> getRewardInfo(@RequestParam Integer rewardId) {

        return ResponseEntity.ok(rewardService.getById(rewardId));
    }

    @PutMapping("/info")
    public ResponseEntity<String> putReward(@RequestBody Reward reward) {
        Reward rewardUpdate = rewardService.getById(reward.getRewardId());
        rewardUpdate.setRewardName(reward.getRewardName());
        rewardUpdate.setRewardPicture(reward.getRewardPicture());
        rewardUpdate.setPrice(reward.getPrice());
        rewardUpdate.setRewardDescription(reward.getRewardDescription());
        rewardUpdate.setLastUpdateAt(Date.from(LocalDate.now().atStartOfDay(ZoneId.systemDefault()).toInstant()));
        rewardService.updateById(rewardUpdate);

        return ResponseEntity.ok("");
    }

    @PostMapping(value = "/completion")
    public ResponseEntity<String> putRewardCompletion(@RequestBody Reward reward) {
        Reward rewardUpdate = rewardService.getById(reward.getRewardId());
        rewardUpdate.setCompletedBy(reward.getCompletedBy());
        rewardUpdate.setCompletedAt(reward.getCompletedAt());
        rewardUpdate.setResourceId(reward.getResourceId());
        rewardService.updateById(rewardUpdate);

        return ResponseEntity.ok("");
    }


}
