package com.g9.g9backend.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.UpdateWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.g9.g9backend.pojo.*;
import com.g9.g9backend.pojo.DTO.*;
import com.g9.g9backend.pojo.Thread;
import com.g9.g9backend.service.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;

/**
 * UserController 用户控制器类，处理与用户相关的请求
 *
 * @author hcy
 */
@RestController
@RequestMapping("/user")
public class UserController {

    private final UserService userService;

    private final InvitationService invitationService;

    private final SubscriptionService subscriptionService;

    private final SearchHistoryService searchHistoryService;

    private final RewardService rewardService;

    private final UserCollectionService userCollectionService;

    private final UserUploadService userUploadService;

    private final UserPurchaseService userPurchaseService;

    private final ResourceService resourceService;

    private final ThreadService threadService;

    public UserController(UserService userService, InvitationService invitationService, SubscriptionService subscriptionService, SearchHistoryService searchHistoryService, RewardService rewardService, UserCollectionService userCollectionService, UserUploadService userUploadService, UserPurchaseService userPurchaseService, ResourceService resourceService, ThreadService threadService) {
        this.userService = userService;
        this.invitationService = invitationService;
        this.subscriptionService = subscriptionService;
        this.searchHistoryService = searchHistoryService;
        this.rewardService = rewardService;
        this.userCollectionService = userCollectionService;
        this.userUploadService = userUploadService;
        this.userPurchaseService = userPurchaseService;
        this.resourceService = resourceService;
        this.threadService = threadService;
    }


    private final Logger logger = LoggerFactory.getLogger(UserController.class);

    /**
     * 用户注册
     *
     * @param registerDTO 用户注册
     * @return 注册结果
     */
    @PostMapping("/register")
    public ResponseEntity<String> register(@RequestBody RegisterDTO registerDTO) {
        String username = registerDTO.getUsername();
        String password = registerDTO.getPassword();
        String invitationCode = registerDTO.getInvitationCode();
        logger.info("Register request received for account: {}", username);

        // 根据用户名查询该用户名是否已存在
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("username", username);
        User userCheck = userService.getOne(userQuery);

        if (userCheck != null) {
            // 用户名重复
            logger.warn("Registration attempt failed. Account already exists: {}", username);
            return ResponseEntity.status(407).body("");
        }

        // 查询邀请码是否存在
        QueryWrapper<Invitation> invitationQuery = new QueryWrapper<>();
        invitationQuery.eq("invitation_code", invitationCode);
        Invitation invitation = invitationService.getOne(invitationQuery);

        if (invitation == null) {
            // 邀请码不存在
            logger.info("The invitation code does not exist: {}", invitationCode);
            return ResponseEntity.status(409).body("");
        } else if (invitation.getInviteeId() != 0) {
            // 邀请码已被使用
            logger.info("The invitation code has been used: {}", invitationCode);
            return ResponseEntity.status(410).body("");
        }
        // 注册
        // 添加新用户
        User user = new User();
        user.setUsername(username);
        user.setPassword(password);
        userService.save(user);

        // 设置该邀请码已被使用
        User userGetId = userService.getOne(userQuery);
        int newUserId = userGetId.getUserId();

        UpdateWrapper<Invitation> invitationUpdate = new UpdateWrapper<>();
        invitationUpdate.eq("invitation_code", invitationCode).set("invitee_id", newUserId);
        invitationService.update(invitationUpdate);

        // 生成五个邀请码并分配给新用户
        String[] invitationCodes = invitationService.generateInvitationCode();

        for (String code : invitationCodes) {
            Invitation newInvitation = new Invitation();
            newInvitation.setInvitationCode(code);
            newInvitation.setUserId(newUserId);
            invitationService.save(newInvitation);
        }

        logger.info("User registered successfully: {}", username);
        return ResponseEntity.ok("");
    }

    /**
     * 用户登录
     *
     * @param user 登录信息
     * @return 登录结果
     */
    @PostMapping("/login")
    public ResponseEntity<String> login(@RequestBody User user) {
        String username = user.getUsername();
        String password = user.getPassword();
        logger.info("Login attempt for account: {}", username);

        // 根据用户名查询该用户名是否已存在
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("username", username);
        User userCheck = userService.getOne(userQuery);

        if (userCheck == null) {
            // 用户名不存在
            logger.warn("Login failed. User not found: {}", username);
            return ResponseEntity.status(406).body("");
        } else {
            if (userCheck.getPassword().equals(password)) {
                return ResponseEntity.ok("");
            } else {
                // 密码错误
                logger.warn("Login failed. Incorrect password for account: {}", username);
                return ResponseEntity.status(408).body("");
            }
        }
    }

    /**
     * 关注
     *
     * @param subscription 关注信息
     * @return 关注结果
     */
    @PostMapping("/subscription")
    public ResponseEntity<String> subscription(@RequestBody Subscription subscription) {
        subscriptionService.save(subscription);
        return ResponseEntity.ok("");
    }

    /**
     * 注销账户
     *
     * @param userId   用户id
     * @param password 密码
     * @return 注销结果
     */
    @DeleteMapping
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public ResponseEntity<String> userDelete(@RequestParam int userId, @RequestParam String password) {
        logger.warn("Delete user with id: {}", userId);
        // 根据用户id查询该用户
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("user_id", userId);
        User userCheck = userService.getOne(userQuery);

        if (userCheck.getPassword().equals(password)) {
            // 注销账户（自动清除与该用户相关的记录，注意会回收分配给该用户的邀请码）
            userService.remove(userQuery);

            return ResponseEntity.noContent().build();
        } else {
            logger.warn("Delete failed. Incorrect password for account: {}", password);
            return ResponseEntity.status(408).body("");
        }
    }

    /**
     * 删除用户搜索历史
     *
     * @param searchHistoryId 搜索历史id
     * @return 删除用户搜索历史结果
     */
    @DeleteMapping("/search")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public ResponseEntity<String> searchDelete(@RequestParam int searchHistoryId) {
        searchHistoryService.removeById(searchHistoryId);
        return ResponseEntity.noContent().build();
    }

    /**
     * 取消关注
     *
     * @param userId     被关注者id
     * @param followerId 关注被人者id
     * @return 取消关注结果
     */
    @DeleteMapping("/subscription")
    @ResponseStatus(HttpStatus.NO_CONTENT)
    public ResponseEntity<String> subscriptionDelete(@RequestParam int userId, @RequestParam int followerId) {
        QueryWrapper<Subscription> subscriptionQuery = new QueryWrapper<>();
        subscriptionQuery.eq("user_id", userId).eq("follower_id", followerId);
        subscriptionService.remove(subscriptionQuery);
        return ResponseEntity.noContent().build();
    }


    /**
     * 获取用户ID
     *
     * @param username 用户名
     * @param password 密码
     * @return 获取邀请码结果
     */
    @GetMapping("/getId")
    public ResponseEntity<Integer> getUserId(@RequestParam String username, @RequestParam String password) {
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("username", username).eq("password", password);
        User user = userService.getOne(userQuery);
        return ResponseEntity.ok(user.getUserId());
    }

    /**
     * 获取邀请码
     *
     * @param userId 用户id
     * @return 获取邀请码结果
     */
    @GetMapping("/invitation-code")
    public ResponseEntity<GetInvitationCodeDTO> getInvitationCode(@RequestParam int userId) {
        QueryWrapper<Invitation> invitationQuery = new QueryWrapper<>();
        invitationQuery.eq("user_id", userId).eq("invitee_id", 0); // 获取未被使用过的分配给该用户的邀请码
        invitationQuery.select("invitation_code");
        List<Invitation> invitationList = invitationService.list(invitationQuery);
        GetInvitationCodeDTO getInvitationCodeDTO = new GetInvitationCodeDTO(invitationList);
        return ResponseEntity.ok(getInvitationCodeDTO);
    }

    /**
     * 获取用户信息
     *
     * @param userId 用户id
     * @return 获取用户信息结果
     */
    @GetMapping("/info")
    public ResponseEntity<User> getUserInfo(@RequestParam int userId) {
        User user = userService.getById(userId);
        return ResponseEntity.ok(user);
    }

    /**
     * 获取用户悬赏
     *
     * @param userId     用户id
     * @param pageNumber 页数
     * @param rows       行数
     * @return 获取用户悬赏结果
     */
    @GetMapping("/reward")
    public ResponseEntity<GetUserRewardDTO> getUserReward(@RequestParam int userId, @RequestParam int pageNumber, @RequestParam int rows) {
        IPage<Reward> page = new Page<>(pageNumber, rows);
        QueryWrapper<Reward> rewardQuery = new QueryWrapper<>();
        rewardQuery.eq("user_id", userId);
        IPage<Reward> rewardPage = rewardService.page(page, rewardQuery);
        List<Reward> rewardList = rewardPage.getRecords();
        long total = rewardPage.getTotal();
        long pages = rewardPage.getPages();
        long current = rewardPage.getCurrent();
        long size = rewardPage.getSize();

        GetUserRewardDTO getUserRewardDTO = new GetUserRewardDTO(rewardList, total, pages, current, size);
        return ResponseEntity.ok(getUserRewardDTO);
    }

    /**
     * 获取用户收藏资源
     *
     * @param userId     用户id
     * @param pageNumber 页数
     * @param rows       行数
     * @return 获取用户收藏结果
     */
    @GetMapping("/collection")
    public ResponseEntity<GetUserResourceListDTO> getUserCollection(@RequestParam int userId, @RequestParam int pageNumber, @RequestParam int rows) {
        IPage<UserCollection> page = new Page<>(pageNumber, rows);
        QueryWrapper<UserCollection> userCollectionQuery = new QueryWrapper<>();
        userCollectionQuery.eq("user_id", userId);
        IPage<UserCollection> userCollectionPage = userCollectionService.page(page, userCollectionQuery);
        List<UserCollection> resourceList = userCollectionPage.getRecords();
        List<Resource> collectionList = new ArrayList<>();
        for (UserCollection userCollection : resourceList) {
            Resource resource = resourceService.getById(userCollection.getResourceId());
            collectionList.add(resource);
        }
        long total = userCollectionPage.getTotal();
        long pages = userCollectionPage.getPages();
        long current = userCollectionPage.getCurrent();
        long size = userCollectionPage.getSize();

        GetUserResourceListDTO getUserResourceListDTO = new GetUserResourceListDTO(collectionList, total, pages, current, size);
        return ResponseEntity.ok(getUserResourceListDTO);
    }

    /**
     * 获取用户上传资源
     *
     * @param userId     用户id
     * @param pageNumber 页数
     * @param rows       行数
     * @return 获取用户上传资源结果
     */
    @GetMapping("/upload")
    public ResponseEntity<GetUserResourceListDTO> getUserUpload(@RequestParam int userId, @RequestParam int pageNumber, @RequestParam int rows) {
        IPage<UserUpload> page = new Page<>(pageNumber, rows);
        QueryWrapper<UserUpload> userUploadQuery = new QueryWrapper<>();
        userUploadQuery.eq("user_id", userId);
        IPage<UserUpload> userUploadPage = userUploadService.page(page, userUploadQuery);
        List<UserUpload> resourceList = userUploadPage.getRecords();
        List<Resource> uploadList = new ArrayList<>();
        for (UserUpload userUpload : resourceList) {
            Resource resource = resourceService.getById(userUpload.getResourceId());
            uploadList.add(resource);
        }
        long total = userUploadPage.getTotal();
        long pages = userUploadPage.getPages();
        long current = userUploadPage.getCurrent();
        long size = userUploadPage.getSize();

        GetUserResourceListDTO getUserResourceListDTO = new GetUserResourceListDTO(uploadList, total, pages, current, size);
        return ResponseEntity.ok(getUserResourceListDTO);
    }

    /**
     * 获取用户购买资源
     *
     * @param userId     用户id
     * @param pageNumber 页数
     * @param rows       行数
     * @return 获取用户购买资源结果
     */
    @GetMapping("/purchase")
    public ResponseEntity<GetUserResourceListDTO> getUserPurchase(@RequestParam int userId, @RequestParam int pageNumber, @RequestParam int rows) {
        IPage<UserPurchase> page = new Page<>(pageNumber, rows);
        QueryWrapper<UserPurchase> userPurchaseQuery = new QueryWrapper<>();
        userPurchaseQuery.eq("user_id", userId);
        IPage<UserPurchase> userPurchasePage = userPurchaseService.page(page, userPurchaseQuery);
        List<UserPurchase> resourceList = userPurchasePage.getRecords();
        List<Resource> purchaseList = new ArrayList<>();
        for (UserPurchase userPurchase : resourceList) {
            Resource resource = resourceService.getById(userPurchase.getResourceId());
            purchaseList.add(resource);
        }
        long total = userPurchasePage.getTotal();
        long pages = userPurchasePage.getPages();
        long current = userPurchasePage.getCurrent();
        long size = userPurchasePage.getSize();

        GetUserResourceListDTO getUserResourceListDTO = new GetUserResourceListDTO(purchaseList, total, pages, current, size);
        return ResponseEntity.ok(getUserResourceListDTO);
    }

    /**
     * 获取用户搜索历史
     *
     * @param userId 用户id
     * @return 获取用户搜索历史结果
     */
    @GetMapping("/search")
    public ResponseEntity<GetUserSearchHistoryDTO> getUserSearchHistory(@RequestParam int userId) {
        QueryWrapper<SearchHistory> SearchHistoryQuery = new QueryWrapper<>();
        SearchHistoryQuery.eq("user_id", userId).orderByDesc("search_history_id").last("LIMIT 10");
        List<SearchHistory> searchHistoryList = searchHistoryService.list(SearchHistoryQuery);

        GetUserSearchHistoryDTO getUserSearchHistoryDTO = new GetUserSearchHistoryDTO(searchHistoryList);
        return ResponseEntity.ok(getUserSearchHistoryDTO);
    }

    /**
     * 获取用户发布贴子
     *
     * @param userId     用户id
     * @param pageNumber 页数
     * @param rows       行数
     * @return 获取用户发布贴子结果
     */
    @GetMapping("/thread")
    public ResponseEntity<GetUserThreadListDTO> getUserThread(@RequestParam int userId, @RequestParam int pageNumber, @RequestParam int rows) {
        IPage<Thread> page = new Page<>(pageNumber, rows);
        QueryWrapper<Thread> threadQuery = new QueryWrapper<>();
        threadQuery.eq("user_id", userId);
        IPage<Thread> threadPage = threadService.page(page, threadQuery);
        List<Thread> threadList = threadPage.getRecords();
        long total = threadPage.getTotal();
        long pages = threadPage.getPages();
        long current = threadPage.getCurrent();
        long size = threadPage.getSize();

        GetUserThreadListDTO getUserThreadListDTO = new GetUserThreadListDTO(threadList, total, pages, current, size);
        return ResponseEntity.ok(getUserThreadListDTO);
    }

    /**
     * 获取用户数据
     *
     * @param userId 用户id
     * @return 获取用户数据结果
     */
    @GetMapping("/data")
    public ResponseEntity<GetUserDataDTO> getUserData(@RequestParam int userId) {
        User user = userService.getById(userId);
        int subscriberCount = user.getSubscriberCount();
        int uploadAmount = user.getUploadAmount();

        QueryWrapper<UserUpload> UserUploadQuery = new QueryWrapper<>();
        UserUploadQuery.eq("user_id", userId);
        List<UserUpload> UserUpload = userUploadService.list(UserUploadQuery);
        int beDownloadedAmount = 0; // 上传资源被下载量
        float[] seedPercentageList = new float[4]; // 上传资源类型百分比，0-1区间的小数
        int resourcePackAmount = 0;
        int modAmount = 0;
        int modPackAmount = 0;
        int mapAmount = 0;
        for (UserUpload userUpload : UserUpload) {
            Resource resource = resourceService.getById(userUpload.getResourceId());
            beDownloadedAmount += resource.getDownloads();
            switch (resource.getClassify()) {
                case "resourcePack":
                    resourcePackAmount++;
                    break;
                case "mod":
                    modAmount++;
                    break;
                case "modPack":
                    modPackAmount++;
                    break;
                case "map":
                    mapAmount++;
                    break;
            }
        }
        int totalAmount = resourcePackAmount + modAmount + modPackAmount + mapAmount;
        seedPercentageList[0] = (float) resourcePackAmount / totalAmount;
        seedPercentageList[1] = (float) modAmount / totalAmount;
        seedPercentageList[2] = (float) modPackAmount / totalAmount;
        seedPercentageList[3] = (float) mapAmount / totalAmount;

        GetUserDataDTO getUserDataDTO = new GetUserDataDTO(subscriberCount, uploadAmount, beDownloadedAmount, seedPercentageList);
        return ResponseEntity.ok(getUserDataDTO);
    }

    /**
     * 获取关注列表
     *
     * @param userId 用户id
     * @return 获取关注列表结果
     */
    @GetMapping("/subscriber")
    public ResponseEntity<GetUserDTO> getUserSubscriber(@RequestParam int userId) {
        QueryWrapper<Subscription> subscriptionQuery = new QueryWrapper<>();
        subscriptionQuery.eq("follower_id", userId);
        List<Subscription> subscriptionList = subscriptionService.list(subscriptionQuery);

        List<User> subscriberList = new ArrayList<>();
        for (Subscription subscription : subscriptionList) {
            User user = userService.getById(subscription.getUserId());
            subscriberList.add(user);
        }

        GetUserDTO getUserDTO = new GetUserDTO(subscriberList);
        return ResponseEntity.ok(getUserDTO);
    }

    /**
     * 获取粉丝列表
     *
     * @param userId 用户id
     * @return 获取粉丝列表结果
     */
    @GetMapping("/follower")
    public ResponseEntity<GetUserDTO> getUserFollower(@RequestParam int userId) {
        QueryWrapper<Subscription> subscriptionQuery = new QueryWrapper<>();
        subscriptionQuery.eq("user_id", userId);
        List<Subscription> subscriptionList = subscriptionService.list(subscriptionQuery);

        List<User> subscriberList = new ArrayList<>();
        for (Subscription subscription : subscriptionList) {
            User user = userService.getById(subscription.getUserId());
            subscriberList.add(user);
        }

        GetUserDTO getUserDTO = new GetUserDTO(subscriberList);
        return ResponseEntity.ok(getUserDTO);
    }

    /**
     * 修改用户签名
     *
     * @param user 用户信息
     * @return 用户签名修改结果
     */
    @PutMapping("/signature")
    public ResponseEntity<String> modifySignature(@RequestBody User user) {
        UpdateWrapper<User> userUpdate = new UpdateWrapper<>();
        userUpdate.eq("user_id", user.getUserId()).set("signature", user.getSignature());
        userService.update(userUpdate);

        return ResponseEntity.ok("");
    }

    /**
     * 修改用户密码
     *
     * @param modifyPasswordDTO 用户信息
     * @return 用户密码修改结果
     */
    @PutMapping("/password")
    public ResponseEntity<String> modifyPassword(@RequestBody ModifyPasswordDTO modifyPasswordDTO) {
        // 检查密码是否正确
        QueryWrapper<User> userQuery = new QueryWrapper<>();
        userQuery.eq("user_id", modifyPasswordDTO.getUserId());
        User user = userService.getOne(userQuery);
        if (user.getPassword().equals(modifyPasswordDTO.getPassword())) {
            UpdateWrapper<User> userUpdate = new UpdateWrapper<>();
            userUpdate.eq("user_id", modifyPasswordDTO.getUserId()).set("password", modifyPasswordDTO.getNewPassword());
            userService.update(userUpdate);
            return ResponseEntity.ok("");
        } else {
            // 密码错误
            logger.warn("Modify failed. Incorrect password for userID: {}", modifyPasswordDTO.getUserId());
            return ResponseEntity.status(408).body("");
        }
    }

    /**
     * 修改用户头像
     *
     * @param user 用户信息
     * @return 用户头像修改结果
     */
    @PutMapping("/avatar")
    public ResponseEntity<String> modifyAvatar(@RequestBody User user) {
        UpdateWrapper<User> userUpdate = new UpdateWrapper<>();
        userUpdate.eq("user_id", user.getUserId()).set("avatar", user.getAvatar());
        userService.update(userUpdate);

        return ResponseEntity.ok("");
    }
}