package com.g9.g9backend.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.g9.g9backend.pojo.*;
import com.g9.g9backend.pojo.DTO.*;
import com.g9.g9backend.service.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

public class UserControllerTest {

    private MockMvc mockMvc;

    @InjectMocks
    private UserController userController;

    @Mock
    private UserService userService;

    @Mock
    private InvitationService invitationService;

    @Mock
    private SubscriptionService subscriptionService;

    @Mock
    private SearchHistoryService searchHistoryService;

    @Mock
    private RewardService rewardService;

    @Mock
    private UserCollectionService userCollectionService;

    @Mock
    private UserUploadService userUploadService;

    @Mock
    private UserPurchaseService userPurchaseService;

    @Mock
    private ResourceService resourceService;

    private final ObjectMapper objectMapper = new ObjectMapper();

    @BeforeEach
    public void setup() {
        MockitoAnnotations.openMocks(this);
        mockMvc = MockMvcBuilders.standaloneSetup(userController).build();
    }

    // 注册
    @Test
    public void testRegister_success() throws Exception {
        /*
        注意单元测试中（使用Mockito ），当多次为同一个方法设置不同的返回值时，后面的设置会覆盖前面的设置
        可见单元测试在执行时，本质上在跑实际接口实现，当遇到被mock模拟的方法时，会来测试代码里找对应的when...thenReturn，直接看最后一个设置，所以当测试方法时，测试代码里写的和实际接口的顺序上不是一一对应
        而是测试代码里写B C A，实际接口代码里写A B C，当实际代码跑到A时去测试代码里从后往前找到A，而不是看测试代码里第一个是不是A若是再用的逻辑
        所以如果写成如下形式：
        when(userService.getOne(any())).thenReturn(null);
        ....其他代码
        when(userService.getOne(any())).thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0));
        则下面的设置会覆盖上面的设置，导致无论什么时候调用该方法，返回的都是非空对象
        所以应该写成如下形式：
        when(userService.getOne(any()))
            .thenReturn(null)          // 第一次调用返回null
            .thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0)); // 第二次调用返回非空对象
        */

        when(userService.getOne(any()))
                .thenReturn(null)          // 第一次调用返回null，表示用户名不是已存在的
                .thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0)); // 第二次调用返回非空对象，表示获取添加后的用户信息
        // 邀请码存在且未被使用
        when(invitationService.getOne(any())).thenReturn(new Invitation("6RCA7Y8J", 1, 0));
        // 添加新用户
        when(userService.save(any())).thenReturn(true);
        // 设置该邀请码被新添加的新用户使用
        when(invitationService.update(any())).thenReturn(true);
        // 获取五个新邀请码
        when(invitationService.generateInvitationCode()).thenReturn(new String[5]);
        // 分配给新用户邀请码
        when(invitationService.save(any())).thenReturn(true);
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().isOk());
    }

    @Test
    public void testRegister_userExists() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0));
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(407));
    }

    @Test
    public void testRegister_invitationNotExist() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(null);
        // 邀请码不存在
        when(invitationService.getOne(any())).thenReturn(null);
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(409));
    }

    @Test
    public void testRegister_invitationUsed() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(null);
        // 邀请码存在但已被使用
        when(invitationService.getOne(any())).thenReturn(new Invitation("6RCA7Y8J", 1, 2));
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(410));
    }

    // 登录
    @Test
    public void testLogin_success() throws Exception {
        // 设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("123");

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().isOk());
    }

    @Test
    public void testLogin_userNotFound() throws Exception {
        //设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("123");

        when(userService.getOne(any())).thenReturn(null);

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().is(406));
    }

    @Test
    public void testLogin_wrongPassword() throws Exception {
        // 设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("wrongPassword");

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().is(408));
    }

    // 关注
    @Test
    public void testSubscription() throws Exception {
        when(subscriptionService.save(any())).thenReturn(true);

        mockMvc.perform(post("/user/subscription")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new Subscription(1, 2))))
                .andExpect(status().isOk());
    }

    // 注销
    @Test
    public void testUserDelete_success() throws Exception {

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(delete("/user")
                        .param("userId", "1")
                        .param("password", "123"))
                .andExpect(status().isNoContent());
    }

    @Test
    public void testUserDelete_wrongPassword() throws Exception {

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(delete("/user")
                        .param("userId", "1")
                        .param("password", "wrong"))
                .andExpect(status().is(408));
    }

    // 删除搜索历史
    @Test
    public void testSearchDelete() throws Exception {

        when(searchHistoryService.removeById(anyInt())).thenReturn(true);

        mockMvc.perform(delete("/user/search")
                        .param("searchHistoryId", "1"))
                .andExpect(status().isNoContent());
    }

    // 取消关注
    @Test
    public void testSubscriptionDelete() throws Exception {

        when(subscriptionService.remove(any())).thenReturn(true);

        mockMvc.perform(delete("/user/subscription")
                        .param("userId", "1")
                        .param("followerId", "2"))
                .andExpect(status().isNoContent());
    }

    // 获取用户ID
    @Test
    public void testGetUserId() throws Exception {

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(get("/user/getId")
                        .param("username", "hcy")
                        .param("password", "123"))
                .andExpect(status().isOk());
    }

    // 获取邀请码
    @Test
    public void testGetInvitationCode() throws Exception {

        List<Invitation> invitationList = new ArrayList<>();
        invitationList.add(new Invitation("1RCA7Y8J", 0, 0));
        invitationList.add(new Invitation("2RCA7Y8J", 0, 0));
        invitationList.add(new Invitation("3RCA7Y8J", 0, 0));
        invitationList.add(new Invitation("4RCA7Y8J", 0, 0));
        invitationList.add(new Invitation("5RCA7Y8J", 0, 0));

        QueryWrapper<Invitation> invitationQuery = new QueryWrapper<>();
        invitationQuery.eq("user_id", 1).eq("invitee_id", 0);
        invitationQuery.select("invitation_code");

        when(invitationService.list(invitationQuery)).thenReturn(invitationList); //这里用any()无法识别，因为list有多种参数可能

        mockMvc.perform(get("/user/invitation-code")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 获取用户信息
    @Test
    public void testGetUserInfo() throws Exception {

        when(userService.getById(anyInt())).thenReturn(new User(1, "user", "pass", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(get("/user/info")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 获取悬赏
    @Test
    public void testGetUserReward() throws Exception {

        when(rewardService.page(any(), any())).thenReturn(new Page<>(1, 2));

        mockMvc.perform(get("/user/reward")
                        .param("userId", "1")
                        .param("pageNumber", "2")
                        .param("rows", "2"))
                .andExpect(status().isOk());
    }

    // 获取收藏
    @Test
    public void testGetUserCollection() throws Exception {

        when(userCollectionService.page(any(), any())).thenReturn(new Page<>(1, 2));
        when(resourceService.getById(anyInt())).thenReturn(new Resource(1, "a", null, null, null, new Date(), new Date(), 100, 0, 0, 0, 0, 0, "mod"));

        mockMvc.perform(get("/user/collection")
                        .param("userId", "1")
                        .param("pageNumber", "1")
                        .param("rows", "2"))
                .andExpect(status().isOk());
    }

    // 获取上传
    @Test
    public void testGetUserUpload() throws Exception {

        when(userUploadService.page(any(), any())).thenReturn(new Page<>(1, 2));
        when(resourceService.getById(anyInt())).thenReturn(new Resource(1, "a", null, null, null, new Date(), new Date(), 100, 0, 0, 0, 0, 0, "mod"));

        mockMvc.perform(get("/user/upload")
                        .param("userId", "1")
                        .param("pageNumber", "1")
                        .param("rows", "2"))
                .andExpect(status().isOk());
    }

    // 获取购买
    @Test
    public void testGetUserPurchase() throws Exception {

        when(userPurchaseService.page(any(), any())).thenReturn(new Page<>(1, 2));
        when(resourceService.getById(anyInt())).thenReturn(new Resource(1, "a", null, null, null, new Date(), new Date(), 100, 0, 0, 0, 0, 0, "mod"));

        mockMvc.perform(get("/user/purchase")
                        .param("userId", "1")
                        .param("pageNumber", "1")
                        .param("rows", "2"))
                .andExpect(status().isOk());
    }

    // 获取搜索历史
    @Test
    public void testGetUserSearchHistory() throws Exception {

        List<SearchHistory> searchHistoryList = new ArrayList<>();

        QueryWrapper<SearchHistory> searchHistoryQuery = new QueryWrapper<>();
        searchHistoryQuery.eq("user_id", 1).orderByDesc("search_history_id").last("LIMIT 10");

        when(searchHistoryService.list(searchHistoryQuery)).thenReturn(searchHistoryList);

        mockMvc.perform(get("/user/search")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 获取用户数据
    @Test
    public void testGetUserData() throws Exception {

        when(userService.getById(1)).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        List<UserUpload> UserUpload = new ArrayList<>();
        QueryWrapper<UserUpload> userUploadQuery = new QueryWrapper<>();
        userUploadQuery.eq("user_id", 1);
        when(userUploadService.list(userUploadQuery)).thenReturn(UserUpload);

        when(resourceService.getById(anyInt())).thenReturn(new Resource());

        mockMvc.perform(get("/user/data")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 获取关注列表
    @Test
    public void testGetUserSubscriber() throws Exception {

        QueryWrapper<Subscription> subscriptionQuery = new QueryWrapper<>();
        subscriptionQuery.eq("follower_id", 1);
        List<Subscription> subscriptionList = new ArrayList<>();

        when(subscriptionService.list(subscriptionQuery)).thenReturn(subscriptionList);
        when(userService.getById(1)).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(get("/user/subscriber")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 获取粉丝列表
    @Test
    public void testGetUserFollower() throws Exception {

        QueryWrapper<Subscription> subscriptionQuery = new QueryWrapper<>();
        subscriptionQuery.eq("user_id", 1);
        List<Subscription> subscriptionList = new ArrayList<>();

        when(subscriptionService.list(subscriptionQuery)).thenReturn(subscriptionList);
        when(userService.getById(1)).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(get("/user/follower")
                        .param("userId", "1"))
                .andExpect(status().isOk());
    }

    // 修改签名
    @Test
    public void testModifySignature() throws Exception {

        User user = new User();
        user.setUserId(1);
        user.setSignature("New Signature");

        when(userService.update(any())).thenReturn(true);

        mockMvc.perform(put("/user/signature")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().isOk());
    }

    // 修改密码
    @Test
    public void testModifyPassword_success() throws Exception {

        // 密码正确
        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "oldPass", null, 0, 0, null, 0, 0, 0));
        when(userService.update(any())).thenReturn(true);

        mockMvc.perform(put("/user/password")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new ModifyPasswordDTO(1, "oldPass", "newPass"))))
                .andExpect(status().isOk());
    }

    @Test
    public void testModifyPassword_wrongPassword() throws Exception {

        // 密码错误
        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "oldPass", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(put("/user/password")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new ModifyPasswordDTO(1, "wrongPassword", "newPass"))))
                .andExpect(status().is(408));
    }

    // 修改头像
    @Test
    public void testModifyAvatar() throws Exception {

        // 设置请求参数
        User user = new User();
        user.setUserId(1);
        user.setAvatar("avatar.png");

        when(userService.update(any())).thenReturn(true);

        mockMvc.perform(put("/user/avatar")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().isOk());
    }
}