package com.g9.g9backend.controller;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.g9.g9backend.pojo.*;
import com.g9.g9backend.pojo.DTO.*;
import com.g9.g9backend.service.*;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;

import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.when;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.*;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.*;

public class UserControllerTest {

    private MockMvc mockMvc;

    @InjectMocks
    private UserController userController;

    @Mock
    private UserService userService;

    @Mock
    private InvitationService invitationService;

    @Mock
    private SubscriptionService subscriptionService;

    private final ObjectMapper objectMapper = new ObjectMapper();

    @BeforeEach
    public void setup() {
        MockitoAnnotations.openMocks(this);
        mockMvc = MockMvcBuilders.standaloneSetup(userController).build();
    }

    // 注册
    @Test
    public void testRegister_success() throws Exception {
        /*
        注意单元测试中（使用Mockito ），当多次为同一个方法设置不同的返回值时，后面的设置会覆盖前面的设置
        可见单元测试在执行时，本质上在跑实际接口实现，当遇到被mock模拟的方法时，会来测试代码里找对应的when...thenReturn，直接看最后一个设置，所以当测试方法时，测试代码里写的和实际接口的顺序上不是一一对应
        而是测试代码里写B C A，实际接口代码里写A B C，当实际代码跑到A时去测试代码里从后往前找到A，而不是看测试代码里第一个是不是A若是再用的逻辑
        所以如果写成如下形式：
        when(userService.getOne(any())).thenReturn(null);
        ....其他代码
        when(userService.getOne(any())).thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0));
        则下面的设置会覆盖上面的设置，导致无论什么时候调用该方法，返回的都是非空对象
        所以应该写成如下形式：
        when(userService.getOne(any()))
            .thenReturn(null)          // 第一次调用返回null
            .thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0)); // 第二次调用返回非空对象
        */

        when(userService.getOne(any()))
                .thenReturn(null)          // 第一次调用返回null，表示用户名不是已存在的
                .thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0)); // 第二次调用返回非空对象，表示获取添加后的用户信息
        // 邀请码存在且未被使用
        when(invitationService.getOne(any())).thenReturn(new Invitation("6RCA7Y8J", 1, 0));
        // 添加新用户
        when(userService.save(any())).thenReturn(true);
        // 设置该邀请码被新添加的新用户使用
        when(invitationService.update(any())).thenReturn(true);
        // 获取五个新邀请码
        when(invitationService.generateInvitationCode()).thenReturn(new String[5]);
        // 分配给新用户邀请码
        when(invitationService.save(any())).thenReturn(true);
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().isOk());
    }

    @Test
    public void testRegister_userExists() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(new User(2, "hcy", "123", null, 0, 0, null, 0, 0, 0));
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(407));
    }

    @Test
    public void testRegister_invitationNotExist() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(null);
        // 邀请码不存在
        when(invitationService.getOne(any())).thenReturn(null);
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(409));
    }

    @Test
    public void testRegister_invitationUsed() throws Exception {
        // 用户名不是已存在的
        when(userService.getOne(any())).thenReturn(null);
        // 邀请码存在但已被使用
        when(invitationService.getOne(any())).thenReturn(new Invitation("6RCA7Y8J", 1, 2));
        mockMvc.perform(post("/user/register")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new RegisterDTO("hcy", "123", "6RCA7Y8J"))))
                .andExpect(status().is(410));
    }

    // 登录
    @Test
    public void testLogin_success() throws Exception {
        // 设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("123");

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().isOk());
    }

    @Test
    public void testLogin_userNotFound() throws Exception {
        //设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("123");

        when(userService.getOne(any())).thenReturn(null);

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().is(406));
    }

    @Test
    public void testLogin_wrongPassword() throws Exception {
        // 设置请求参数
        User user = new User();
        user.setUsername("hcy");
        user.setPassword("wrongPassword");

        when(userService.getOne(any())).thenReturn(new User(1, "hcy", "123", null, 0, 0, null, 0, 0, 0));

        mockMvc.perform(post("/user/login")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(user)))
                .andExpect(status().is(408));
    }

    // 关注
    @Test
    public void testSubscription() throws Exception {
        when(subscriptionService.save(any())).thenReturn(true);

        mockMvc.perform(post("/user/subscription")
                        .contentType(MediaType.APPLICATION_JSON)
                        .content(objectMapper.writeValueAsString(new Subscription(1, 2))))
                .andExpect(status().isOk());
    }
}