'use client';
import React, {useEffect, useRef, useState} from 'react';
import { Button } from "primereact/button";
// 输入框
import { InputText } from "primereact/inputtext";
// 消息提醒
import { Toast } from 'primereact/toast';
// 接口传输
import axios from "axios";
// 页面跳转
import { useRouter } from "next/navigation";
// 密码
import { Password } from 'primereact/password';
// 样式
import './login.scss';

interface UserInfo {
    userId: number;
    username: string;
    avatar: string;
    signature: string;
}

export default function LoginPage() {
    // 路由
    const router = useRouter();
    // 消息提醒
    const toast = useRef<Toast>(null);

    const [userId, setUserId] = useState<number>(0);
    const [loginFormData, setLoginFormData] = useState({
        username: '',
        password: '',
    })

    useEffect(() => {
        storeUserInfo();
    }, [userId]);

    // 使用localStorage存储用户数据
    const storeUserInfo = async () => {
        try {
            const response = await axios.get<UserInfo>(process.env.PUBLIC_URL + `/user/info`, {
                params: { userId: userId }
            });

            localStorage.setItem("user", JSON.stringify({
                Id: response.data.userId,
                name: response.data.username,
                avatar: response.data.avatar,
                signature: response.data.signature,
            }));

        } catch (error) {
            console.error('获取用户信息失败:', error);
            toast.current?.show({ severity: 'error', summary: 'Error', detail: '获取用户信息失败' });
        }
    }

    // 获取userId
    const fetchUserId = async () => {
        try {
            const response = await axios.get(process.env.PUBLIC_URL + `/user/getId`, {
                params: { username: loginFormData.username , password: loginFormData.password }
            });
            setUserId(response.data)
        } catch (error) {
            console.error('获取用户id失败:', error);
            toast.current?.show({ severity: 'error', summary: 'Error', detail: '获取用户id失败' });
        }
    }

    // 处理用户登录
    const handleLogin = async () => {
        try {
            const postData = {
                username: loginFormData.username,
                password: loginFormData.password,
            }
            const response = await axios.post(process.env.PUBLIC_URL + `/user/login`, postData, {
                // eslint-disable-next-line @typescript-eslint/no-unused-vars
                validateStatus: (status) => true // 接受所有状态码
            });

            if (response.status === 200) {
                console.log('用户登录成功');
                toast.current?.show({ severity: 'success', summary: 'Success', detail: '登录成功' });
                // 设置userId
                fetchUserId()
                router.push('/');
            } else if (response.status === 406) {
                console.log('用户名不存在');
                toast.current?.show({ severity: 'error', summary: 'Error', detail: '用户名不存在' });
            } else if (response.status === 408) {
                console.error('密码错误');
                toast.current?.show({ severity: 'error', summary: 'Error', detail: '密码错误' });
            }

        } catch (error) {
            console.error('登录失败:', error);
            toast.current?.show({ severity: 'error', summary: 'Error', detail: '登录失败' });
        }
    }

    return (
        <div className='login-container'>
            <Toast ref={toast}></Toast>
            <h1 className="form-header">登录</h1>
            <div className="form-field">
                <div className="form-field-header">
                    <label htmlFor="username">用户名</label>
                </div>
                <InputText
                    id="username"
                    value={loginFormData.username}
                    onChange={(e) => setLoginFormData(prev => ({
                        ...prev, // 复制顶层所有属性
                        username: e.target.value
                    }))}
                    className="w-full"
                />
            </div>
            <div className="form-field">
                <div className="form-field-header">
                    <label htmlFor="password">密码</label>
                </div>
                <Password
                    id="password"
                    value={loginFormData.password}
                    onChange={(e) => setLoginFormData(prev => ({
                        ...prev,
                        password: e.target.value
                    }))}
                    className="w-full"
                    toggleMask
                />
            </div>
            <div className="form-operation">
                <Button label="确定" onClick={handleLogin} className='form-operation-button'/>
                <Button label="注册" onClick={() => router.push("/user/register")} className='form-operation-button'/>
            </div>
        </div>
    )
};
