package edu.bjtu.groupone.backend.service;

import com.turn.ttorrent.tracker.TrackedPeer;
import com.turn.ttorrent.tracker.TrackedTorrent;
import com.turn.ttorrent.tracker.Tracker;

import java.io.*;
import java.net.InetSocketAddress;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import javax.annotation.PostConstruct;

import edu.bjtu.groupone.backend.config.TrafficAwareTracker;
import edu.bjtu.groupone.backend.domain.entity.AnnounceEvent;
import edu.bjtu.groupone.backend.domain.entity.Torrent;
import edu.bjtu.groupone.backend.service.PeerTrafficService;
import edu.bjtu.groupone.backend.service.TorrentService;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

@Component
public class TrackerStarter {
    private Tracker tracker;
    private final Set<String> loadedTorrentNames = new HashSet<>();

    @Autowired
    private PeerTrafficService peerTrafficService;
    @Autowired
    private TorrentService torrentService;

    @PostConstruct
    public void startTracker() throws Exception {
//        InetSocketAddress trackerAddress = new InetSocketAddress("0.0.0.0", 6969);
//        tracker = new Tracker(trackerAddress);
//        tracker.start();
        InetSocketAddress address = new InetSocketAddress("0.0.0.0", 6969);
        this.tracker = new TrafficAwareTracker(address, peerTrafficService);
        this.tracker.start();
        System.out.println("Tracker started with traffic monitoring");
    }

    @Scheduled(fixedRate = 10000L)
    public void scanTorrentDirectory() throws IOException, NoSuchAlgorithmException {
        List<Torrent> torrents = torrentService.getTorrentList();

        for (Torrent torrent : torrents) {
            String url = torrent.getUrl();
            String fileName = extractFileNameFromUrl(url);

            if (!loadedTorrentNames.contains(fileName)) {
                try {
                    // 创建临时文件（自动带 .torrent 后缀）
                    File tempTorrentFile = File.createTempFile("torrent_", ".torrent");
                    tempTorrentFile.deleteOnExit(); // JVM退出时删除

                    // 下载内容到临时文件
                    try (InputStream in = new URL(url).openStream();
                         OutputStream out = new FileOutputStream(tempTorrentFile)) {
                        in.transferTo(out);
                    }

                    // 加载并注册种子
                    TrackedTorrent trackedTorrent = TrackedTorrent.load(tempTorrentFile);
                    this.tracker.announce(trackedTorrent);
                    loadedTorrentNames.add(fileName);

                    System.out.println("Loaded torrent from OSS (temp file): " + fileName);
                    tempTorrentFile.delete();
                } catch (Exception e) {
                    System.err.println("Failed to load torrent: " + fileName + ", error: " + e.getMessage());
                    e.printStackTrace();
                }
            }
        }
    }


    private String extractFileNameFromUrl(String url) {
        return url.substring(url.lastIndexOf('/') + 1);
    }

@Scheduled(fixedRate = 15000L)
public void collectPeerTraffic() {
    try {
        // 使用反射获取 Tracker 的 private 字段 torrents
        java.lang.reflect.Field torrentsField = Tracker.class.getDeclaredField("torrents");
        torrentsField.setAccessible(true);

        @SuppressWarnings("unchecked")
        Map<?, TrackedTorrent> torrents = (Map<?, TrackedTorrent>) torrentsField.get(tracker);

        for (TrackedTorrent torrent : torrents.values()) {
            for (TrackedPeer peer : torrent.getPeers().values()) {
                if (peer.getUploaded() > 0 || peer.getDownloaded() > 0) {
                    peerTrafficService.recordPeerTraffic(
                            torrent.getHexInfoHash(),
                            peer.getIp(),
                            peer.getPeerId(),
                            peer.getUploaded(),
                            peer.getDownloaded(),
                            System.currentTimeMillis() / 1000,
                            AnnounceEvent.COMPLETED // 实际项目中你可以按需要换成 STARTED, STOPPED 等
                    );
                }
            }
        }
    } catch (Exception e) {
        System.err.println("Failed to collect peer traffic: " + e.getMessage());
        e.printStackTrace();
    }
}
}

