package edu.bjtu.groupone.backend;

import com.turn.ttorrent.tracker.Tracker;
import edu.bjtu.groupone.backend.config.TrafficAwareTracker;
import edu.bjtu.groupone.backend.domain.dto.WorkResponse;
import edu.bjtu.groupone.backend.domain.entity.Category;
import edu.bjtu.groupone.backend.domain.entity.Work;
import edu.bjtu.groupone.backend.mapper.WorkMybatisMapper;
import edu.bjtu.groupone.backend.service.CategoryService;
import edu.bjtu.groupone.backend.service.PeerTrafficService;
import edu.bjtu.groupone.backend.service.WorkService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Arrays;
import java.util.List;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@SpringBootTest
@ExtendWith(MockitoExtension.class)
class WorkServiceTest {

    @Mock
    private WorkMybatisMapper workMybatisMapper;

    @Mock
    private CategoryService categoryService;

    @InjectMocks
    private WorkService workService;

    private Work testWork;
    private Category testCategory;
    private List<Work> works;

    @BeforeEach
    void setUp() {
        testCategory = new Category(1L, "Test Category", null);

        testWork = Work.builder()
                .id(1L)
                .title("Test Work")
                .author("Author")
                .views(100)
                .category(testCategory)
                .build();

        Category category = new Category(1L, "文学艺术", null);
        works = Arrays.asList(
                new Work(1L, "《我的世界》", "张三", 1234, category, "一部关于...", "2023-06-15"),
                new Work(2L, "《你的世界》", "张三", 567, category, "另一部关于...", "2023-06-16")
        );
    }

    @Test
    void addWork_shouldCallMapper() {
        workService.addWork(testWork);
        verify(workMybatisMapper, times(1)).save(testWork);
    }

    @Test
    void deleteWork_shouldCallMapper() {
        workService.deleteWork(1L);
        verify(workMybatisMapper, times(1)).deleteById(1L);
    }

    @Test
    void updateWork_shouldCallMapper() {
        workService.updateWork(testWork);
        verify(workMybatisMapper, times(1)).update(testWork);
    }

    @Test
    void getWorkById_shouldReturnWork() {
        when(workMybatisMapper.findById(1L)).thenReturn(testWork);
        Work result = workService.getWorkById(1L);
        assertEquals(testWork, result);
        verify(workMybatisMapper, times(1)).findById(1L);
    }

    @Test
    void getWorks_shouldReturnPage() {
        Long categoryId = 1L;
        int page = 1; // Fix: Use page = 1 to account for WorkService expecting 1-based indexing
        int size = 10;
        List<Long> categoryIds = Arrays.asList(1L);

        when(categoryService.getAllSubcategoryIds(categoryId)).thenReturn(categoryIds);
        Page<Work> workPage = new PageImpl<>(Arrays.asList(testWork));
        when(workMybatisMapper.findByCategoryIdIn(categoryIds, PageRequest.of(0, size))).thenReturn(workPage);

        Page<WorkResponse> result = workService.getWorks(categoryId, page, size);

        assertEquals(1, result.getContent().size());
        verify(categoryService, times(1)).getAllSubcategoryIds(categoryId);
        verify(workMybatisMapper, times(1)).findByCategoryIdIn(categoryIds, PageRequest.of(0, size));
    }

    @Test
    void getWorksByAuthor_Success() {
        when(workMybatisMapper.findByAuthor("张三")).thenReturn(works);
        List<WorkResponse> result = workService.getWorksByAuthor("张三");
        assertEquals(2, result.size());
        assertEquals("《我的世界》", result.get(0).getTitle());
        assertEquals("《你的世界》", result.get(1).getTitle());
    }

    @Test
    void getWorksByAuthor_Empty() {
        when(workMybatisMapper.findByAuthor(anyString())).thenReturn(List.of());
        List<WorkResponse> result = workService.getWorksByAuthor("李四");
        assertTrue(result.isEmpty());
    }

    @Autowired
    PeerTrafficService peerTrafficService;
    @Test
    public  void test() throws IOException {
        Tracker tracker;
        InetSocketAddress trackerAddress = new InetSocketAddress("0.0.0.0", 6969);
        tracker = new Tracker(trackerAddress);
        tracker.start();
    }
}