// src/main/java/edu/bjtu/groupone/backend/controller/UserController.java
package edu.bjtu.groupone.backend.controller;

import edu.bjtu.groupone.backend.model.Result;
import edu.bjtu.groupone.backend.model.User;
import edu.bjtu.groupone.backend.service.UserService;
import edu.bjtu.groupone.backend.utils.JwtUtils;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

@CrossOrigin
@Tag(name = "用户相关接口") //输入http://localhost:8080/swagger-ui/index.html或http://localhost:8080/v3/api-docs/可查看接口文档
@RestController
@Slf4j
public class UserController {

    @Autowired
    private UserService userService;

    /** 邮箱+密码 登录 */
    @Operation(summary = "用户登录", description = "使用邮箱和密码进行登录，成功后返回 JWT token")
    @PostMapping("/login")
    public Result login(@RequestBody Map<String,String> body) {
        String email = body.get("email");
        String password = body.get("password");
        if (email == null || password == null) {
            return Result.error("参数不完整");
        }
        User u = new User();
        u.setEmail(email);
        u.setPassword(password);
        User user = userService.login(u);
        if (user != null) {
            var payload = Map.<String,Object>of(
                    "id",       user.getUserId(),
                    "username", user.getUsername(),
                    "email",    user.getEmail()
            );
            String token = JwtUtils.generateJwt(payload);
            return Result.success(token);
        }
        return Result.error("邮箱或密码错误");
    }

    /** 注册 */
    @Operation(summary = "用户注册", description = "注册新用户，必须提供用户名、邮箱、验证码、密码和身份证号码")
    @PostMapping("/register")
    public Result register(@RequestBody Map<String,String> body) {
        String username = body.get("username");
        String email    = body.get("email");
        String code     = body.get("verificationCode");
        String pwd      = body.get("password");
        String idNumStr = body.get("identificationNumber");
        if (email==null||code==null||pwd==null||idNumStr==null) {
            return Result.error("参数不完整");
        }
        int idNum = Integer.parseInt(idNumStr);
        if (!userService.verifyCode(email, code)) {
            return Result.error("验证码错误或已过期");
        }
        if (userService.isEmailExists(email)) {
            return Result.error("邮箱已被注册");
        }
        if (username!=null && userService.isUsernameExists(username)) {
            return Result.error("用户名已被占用");
        }
        if (idNumStr == null || idNumStr.isBlank()) {
            return Result.error("身份证号码不能为空");
        }
        if (idNumStr.length() !=8){
            return Result.error("身份证号码长度不正确，应为8位");
        }
        if (userService.isIdentificationNumberExists(idNum)) {
            return Result.error("身份证号码已被注册");
        }

        User newUser = new User();
        newUser.setUsername(username);
        newUser.setEmail(email);
        newUser.setPassword(pwd);
        newUser.setIdentificationNumber(idNum);
        userService.register(newUser);
        return Result.success("注册成功");
    }

    /** 发送注册验证码 */
    @Operation(summary = "发送注册验证码", description = "向指定邮箱发送注册验证码")
    @PostMapping("/sendVerification")
    public Result sendVerification(@RequestBody Map<String,String> p) {
        String email = p.get("email");
        if (email==null) {
            return Result.error("邮箱不能为空");
        }
        userService.sendVerificationCode(email);
        return Result.success("验证码已发送");
    }

    @Operation(summary = "发送重置密码验证码", description = "向指定邮箱发送重置密码的验证码")
    @PostMapping("/sendResetCode")
    public Result sendResetCode(@RequestBody Map<String, String> body) {
        String email = body.get("email");
        userService.sendResetCode(email);
        return Result.success("重置验证码已发送");
    }

    @Operation(summary = "重置密码", description = "根据邮箱和验证码重置密码")
    @PostMapping("/resetPassword")
    public Result resetPassword(@RequestBody Map<String, String> body) {
        String email       = body.get("email");
        String code        = body.get("code");
        String newPassword = body.get("newPassword");
        boolean ok = userService.resetPassword(email, code, newPassword);
        return ok
                ? Result.success("密码已重置")
                : Result.error("验证码错误或已过期");
    }

    /** 前端受保护页面获取当前用户信息 */
    @Operation(summary = "获取当前用户信息", description = "需要提供有效的 JWT token")
    @GetMapping("/api/me")
    public Result me(@RequestHeader("token") String token) {
        var claims = JwtUtils.parseJwt(token);
        if (claims == null) {
            return Result.error("无效的 token");
        }
        var info = Map.<String,Object>of(
                "username", claims.get("username"),
                "token", token
        );
        return Result.success(info);
    }
}
