package edu.bjtu.groupone.backend;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import java.time.LocalDateTime;

import ch.qos.logback.classic.Logger;
import org.slf4j.LoggerFactory;
import edu.bjtu.groupone.backend.mapper.UserMapper;
import edu.bjtu.groupone.backend.mapper.InviteCodeMapper;
import edu.bjtu.groupone.backend.model.User;
import edu.bjtu.groupone.backend.model.InviteCode;
import edu.bjtu.groupone.backend.service.impl.UserServImpl;
import edu.bjtu.groupone.backend.service.impl.InviteCodeServiceImpl;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@ExtendWith(MockitoExtension.class)
public class UserControllerTest {

    // 初始化 Logger，避免空指针
    private static final Logger log = (Logger) LoggerFactory.getLogger(UserControllerTest.class);

    @Mock
    private UserMapper userMapper;

    @Mock
    private InviteCodeMapper inviteCodeMapper;

    @InjectMocks
    private UserServImpl userService;

    @InjectMocks
    private InviteCodeServiceImpl inviteCodeService;

    @Test
    public void testLogin() {
        User loginUser = new User();
        loginUser.setIdentificationNumber(123456);
        loginUser.setPassword("password123");
        User expected = new User();
        expected.setUsername("expectedUsername");

        // 模拟 mapper 返回
        when(userMapper.login(loginUser)).thenReturn(expected);

        // 调用 service 并断言
        User result = userService.login(loginUser);
        assertThat(result).isNotNull();
        assertThat(result.getUsername()).isEqualTo("expectedUsername");
        verify(userMapper, times(1)).login(loginUser);
    }

    @Test
    public void testUseInviteCode() {
        String inviteCode = "TESTCODE";

        // 构造一个有效的 InviteCode，注意设置 expiryTime
        InviteCode dummy = new InviteCode();
        dummy.setCode(inviteCode);
        dummy.setRemainingUses(1);
        // 设置有效期在未来，避免调用 expiryTime.isBefore(...)
        dummy.setExpiryTime(LocalDateTime.now().plusDays(1));

        // Stub 两个 Mapper 方法
        when(inviteCodeMapper.selectByCode(inviteCode)).thenReturn(dummy);
        when(inviteCodeMapper.decrementRemainingUses(inviteCode)).thenReturn(1);

        // 调用 service
        boolean used = inviteCodeService.useInviteCode(inviteCode);

        // 断言和验证
        assertThat(used).isTrue();
        verify(inviteCodeMapper, times(1)).selectByCode(inviteCode);
        verify(inviteCodeMapper, times(1)).decrementRemainingUses(inviteCode);
    }
}
