package edu.bjtu.groupone.backend.api;

import edu.bjtu.groupone.backend.domain.dto.WorkResponse;
import edu.bjtu.groupone.backend.domain.entity.Work;
import edu.bjtu.groupone.backend.service.WorkService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

@RestController
@RequestMapping("/api/works")
@Tag(name = "Work", description = "作品相关接口")
public class WorkController {

    @Autowired
    private WorkService workService;

    @GetMapping
    @Operation(summary = "获取作品列表", description = "可按分区分页查询作品")
    public ResponseEntity<Page<WorkResponse>> getWorks(
            @Parameter(description = "分区 ID（可选）") @RequestParam(required = false) Long categoryId,
            @Parameter(description = "页码，从 1 开始", example = "1") @RequestParam(defaultValue = "1") int page,
            @Parameter(description = "每页数量", example = "20") @RequestParam(defaultValue = "20") int size) {
        return ResponseEntity.ok(workService.getWorks(categoryId, page, size));
    }
    @PostMapping
    @Operation(summary = "创建新作品")
    public ResponseEntity<String> createWork(@RequestBody Work work) {
        workService.addWork(work);
        return ResponseEntity.ok("作品创建成功");
    }

    // 删除作品
    @DeleteMapping("/{id}")
    @Operation(summary = "删除作品")
    public ResponseEntity<String> deleteWork(@PathVariable Long id) {
        workService.deleteWork(id);
        return ResponseEntity.ok("作品删除成功");
    }

    // 更新作品
    @PutMapping("/{id}")
    @Operation(summary = "更新作品信息")
    public ResponseEntity<String> updateWork(@PathVariable Long id, @RequestBody Work work) {
        work.setId(id);
        workService.updateWork(work);
        return ResponseEntity.ok("作品更新成功");
    }

    // 获取单个作品
    @GetMapping("/{id}")
    @Operation(summary = "获取作品详情")
    public ResponseEntity<Work> getWork(@PathVariable Long id) {
        return ResponseEntity.ok(workService.getWorkById(id));
    }
}
