package edu.bjtu.groupone.backend.service.impl;

import edu.bjtu.groupone.backend.mapper.UserMapper;
import edu.bjtu.groupone.backend.domain.entity.User;
import edu.bjtu.groupone.backend.service.UserService;
import edu.bjtu.groupone.backend.utils.EmailUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;

@Service
public class UserServImpl implements UserService {

    @Autowired
    private UserMapper userMapper;
    @Autowired
    private EmailUtil emailUtil;
    private final ConcurrentHashMap<String, String> emailCodes = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, String> resetCodes = new ConcurrentHashMap<>();
    private static final long CODE_EXPIRE_MINUTES = 5;

    @Override
    public User login(User user) {
        return userMapper.login(user);
    }

    @Override
    public boolean isUsernameExists(String username) {
        return userMapper.selectByUsername(username) != null;
    }

    @Override
    public boolean isEmailExists(String email) {
        return userMapper.selectByEmail(email) != null;
    }

    @Override
    public boolean isIdentificationNumberExists(int idNum) {
        return userMapper.countByIdentificationNumber(idNum) > 0;
    }

    @Override
    public void register(User user) {
        // 如果前端未提供用户名，可生成默认
        String username = user.getUsername();
        if (username == null || username.isBlank()) {
            do {
                username = generateDefaultUsername(user.getEmail());
            } while (isUsernameExists(username));
            user.setUsername(username);
        }
        // 设置注册时间
        user.setRegistrationDate(
                LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"))
        );
        // 插入用户记录
        userMapper.insertUser(user);
    }

    @Override
    public void sendVerificationCode(String email) {
        String code = generateRandomCode();
        emailUtil.sendVerificationEmail(email, code);
        long expireAt = System.currentTimeMillis() + CODE_EXPIRE_MINUTES * 60 * 1000;
        emailCodes.put(email, code + "|" + expireAt);
    }

    @Override
    public boolean verifyCode(String email, String inputCode) {
        String stored = emailCodes.get(email);
        if (stored == null) {
            return false;
        }
        String[] parts = stored.split("\\|");
        if (parts.length != 2) {
            return false;
        }
        String code = parts[0];
        long expireTime = Long.parseLong(parts[1]);
        return System.currentTimeMillis() <= expireTime && code.equals(inputCode);
    }


    @Override
    public void sendResetCode(String email) {
        if (!isEmailExists(email)) {
            throw new IllegalArgumentException("该邮箱尚未注册");
        }
        String code = generateRandomCode();
        emailUtil.sendVerificationEmail(email, code);
        long expireAt = System.currentTimeMillis() + CODE_EXPIRE_MINUTES * 60 * 1000;
        resetCodes.put(email, code + "|" + expireAt);
    }

    @Override
    public boolean resetPassword(String email, String code, String newPassword) {
        String stored = resetCodes.get(email);
        if (stored == null) return false;
        String[] p = stored.split("\\|");
        long exp = Long.parseLong(p[1]);
        if (System.currentTimeMillis() > exp || !p[0].equals(code)) {
            return false;
        }
        int cnt = userMapper.updatePasswordByEmail(email, newPassword);
        if (cnt == 1) {
            resetCodes.remove(email);
            return true;
        }
        return false;
    }


    private String generateRandomCode() {
        return String.format("%06d", new Random().nextInt(1_000000));
    }

    private String generateDefaultUsername(String email) {
        String prefix = email != null && email.contains("@")
                ? email.substring(0, email.indexOf("@"))
                : "user";
        return "user_" + prefix + "_" + String.format("%04d", new Random().nextInt(10000));
    }
    @Override
    public void deleteUser(int userId) {
        userMapper.deleteUser(userId);
    }

    @Override
    public void updateUser(User user) {
        userMapper.updateUser(user);
    }

    @Override
    public User getUserById(int userId) {
        return userMapper.selectUserById(userId);
    }

    @Override
    public List<User> getAllUsers() {
        return userMapper.selectAllUsers();
    }
}
