package edu.bjtu.groupone.backend;

import edu.bjtu.groupone.backend.api.WorkController;
import edu.bjtu.groupone.backend.domain.dto.WorkResponse;
import edu.bjtu.groupone.backend.domain.entity.Category;
import edu.bjtu.groupone.backend.domain.entity.Work;
import edu.bjtu.groupone.backend.service.WorkService;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.PageRequest;
import org.springframework.http.ResponseEntity;

import java.util.Collections;

import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

@ExtendWith(MockitoExtension.class)
class WorkControllerTest {

    @Mock
    private WorkService workService;

    @InjectMocks
    private WorkController workController;

    private Work testWork;
    private WorkResponse testResponse;

    @BeforeEach
    void setUp() {
        testWork = new Work();
        testWork.setId(1L);
        testWork.setTitle("Test Work");
        testWork.setAuthor("Test Author");
        testWork.setViews(100);

        // 创建 Category 对象
        Category category = new Category();
        category.setId(1L);
        testWork.setCategory(category);

        // 使用正确的参数创建 WorkResponse
        testResponse = new WorkResponse(
                1L,
                "Test Work",
                "Test Author",
                100,
                1L,
                "Test Description", // 新增 description 参数
                "2023-06-15"
        );
    }

    @Test
    void createWork_shouldReturnSuccess() {
        // 调用
        ResponseEntity<String> response = workController.createWork(testWork);

        // 验证
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("作品创建成功", response.getBody());
        verify(workService, times(1)).addWork(any(Work.class));
    }

    @Test
    void deleteWork_shouldReturnSuccess() {
        // 调用
        ResponseEntity<String> response = workController.deleteWork(1L);

        // 验证
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("作品删除成功", response.getBody());
        verify(workService, times(1)).deleteWork(1L);
    }

    @Test
    void updateWork_shouldReturnSuccess() {
        // 调用
        ResponseEntity<String> response = workController.updateWork(1L, testWork);

        // 验证
        assertEquals(200, response.getStatusCodeValue());
        assertEquals("作品更新成功", response.getBody());
        verify(workService, times(1)).updateWork(any(Work.class));
    }

    @Test
    void getWork_shouldReturnWork() {
        // 模拟
        when(workService.getWorkById(1L)).thenReturn(testWork);

        // 调用
        ResponseEntity<Work> response = workController.getWork(1L);

        // 验证
        assertEquals(200, response.getStatusCodeValue());
        assertEquals(testWork, response.getBody());
        verify(workService, times(1)).getWorkById(1L);
    }

    @Test
    void getWorks_shouldReturnPage() {
        // 创建模拟分页数据
        Page<WorkResponse> mockPage = new PageImpl<>(
                Collections.singletonList(testResponse),
                PageRequest.of(0, 10),
                1
        );

        // 模拟服务层方法
        when(workService.getWorks(eq(null), eq(1), eq(10)))
                .thenReturn(mockPage);

        // 调用
        ResponseEntity<Page<WorkResponse>> response = workController.getWorks(null, 1, 10);

        // 验证
        assertEquals(200, response.getStatusCodeValue());
        assertNotNull(response.getBody());
        assertEquals(1, response.getBody().getTotalElements());
        assertEquals(1, response.getBody().getContent().size());
        assertEquals("Test Work", response.getBody().getContent().get(0).getTitle());
        verify(workService, times(1)).getWorks(null, 1, 10);
    }
}