package edu.bjtu.groupone.backend.api;

import edu.bjtu.groupone.backend.domain.dto.CategoryDTO;
import edu.bjtu.groupone.backend.domain.dto.CategoryCreateRequest;
import edu.bjtu.groupone.backend.domain.entity.Category;
import edu.bjtu.groupone.backend.service.CategoryService;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@RestController
@RequestMapping("/api/category")
@Tag(name = "Category", description = "分区相关接口")
public class CategoryController {

    @Autowired
    private CategoryService categoryService;

    @GetMapping
    @Operation(summary = "获取所有分区（树形结构）", description = "返回所有分区及其子分区")
    public ResponseEntity<List<CategoryDTO>> getCategories() {
        return ResponseEntity.ok(categoryService.getCategoryTree());
    }

    @PostMapping
    @Operation(
        summary = "创建新分区",
        description = "创建一个新的分区，返回创建的分区ID",
        responses = {
            @ApiResponse(
                responseCode = "200",
                description = "创建成功",
                content = @Content(schema = @Schema(implementation = Long.class))
            ),
            @ApiResponse(responseCode = "400", description = "请求参数错误")
        }
    )
    public ResponseEntity<Long> createCategory(
        @Parameter(description = "分区信息", required = true)
        @RequestBody CategoryCreateRequest request
    ) {
        Category category = new Category();
        category.setName(request.getName());
        if (request.getParentId() != null) {
            Category parent = new Category();
            parent.setId(request.getParentId());
            category.setParent(parent);
        }
        return ResponseEntity.ok(categoryService.addCategory(category));
    }

    @DeleteMapping("/{id}")
    @Operation(summary = "删除分区")
    public ResponseEntity<String> deleteCategory(@PathVariable Long id) {
        categoryService.deleteCategory(id);
        return ResponseEntity.ok("分区删除成功");
    }

    @PutMapping("/{id}")
    @Operation(summary = "更新分区信息")
    public ResponseEntity<String> updateCategory(@PathVariable Long id, @RequestBody Category category) {
        category.setId(id);
        categoryService.updateCategory(category);
        return ResponseEntity.ok("分区更新成功");
    }

    @GetMapping("/{id}")
    @Operation(summary = "获取分区详情")
    public ResponseEntity<Category> getCategory(@PathVariable Long id) {
        return ResponseEntity.ok(categoryService.getCategoryById(id));
    }
}