package edu.bjtu.groupone.backend.api;

import edu.bjtu.groupone.backend.domain.entity.Torrent;
import edu.bjtu.groupone.backend.service.TorrentService;
import edu.bjtu.groupone.backend.utils.GetTokenUserId;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ExampleObject;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.util.List;
import java.util.Map;

@Tag(name = "Torrent 管理", description = "种子文件的上传、下载及列表查询")
@RestController
@RequestMapping("/api/torrents")
public class TorrentController {

    private final TorrentService torrentService;

    // 监听器列表应该放在专门的处理类中
    @Autowired
    public TorrentController(TorrentService torrentService) {
        this.torrentService = torrentService;
    }

    @Operation(summary = "上传种子文件", description = "接收前端上传的 .torrent 文件，并保存到服务器和数据库")
    @ApiResponses({
            @ApiResponse(
                    responseCode = "200",
                    description = "上传成功，返回 Torrent 对象",
                    content = @Content(
                            mediaType = "application/json",
                            schema = @Schema(implementation = Torrent.class),
                            examples = @ExampleObject(
                                    name = "上传成功示例",
                                    value = "{\n" +
                                            "  \"id\": 1,\n" +
                                            "  \"infoHash\": \"ABCDEF1234567890ABCDEF1234567890ABCDEF12\",\n" +
                                            "  \"fileName\": \"movie.torrent\",\n" +
                                            "  \"userId\": 1,\n" +
                                            "  \"uploadTime\": \"2025-06-07T16:35:12\"\n" +
                                            "}"
                            )
                    )
            ),
            @ApiResponse(responseCode = "400", description = "文件格式不支持或参数错误")
    })
    @PostMapping("/upload")
    public ResponseEntity<?> uploadTorrent(
            @Parameter(
                    description = "种子文件 (.torrent)",
                    required = true,
                    content = @Content(
                            mediaType = "application/octet-stream",
                            schema = @Schema(type = "string", format = "binary")
                    )
            )
            @RequestParam("file") MultipartFile file,
            @Parameter(description = "上传者用户 ID", required = true, example = "1")
            @RequestParam("id") Long id ) throws Exception {

        Torrent saved = torrentService.uploadTorrent(file, id);
        return ResponseEntity.ok(saved);

    }

    @Operation(summary = "下载种子文件", description = "根据 infoHash 返回对应的种子文件 URL（由 OSS 提供）")
    @ApiResponses({
            @ApiResponse(
                    responseCode = "200",
                    description = "返回 OSS 上的种子文件下载地址",
                    content = @Content(
                            mediaType = "application/json",
                            examples = @ExampleObject(
                                    name = "下载链接示例",
                                    value = "{ \"url\": \"https://your-bucket.oss-cn-region.aliyuncs.com/abc.torrent\" }"
                            )
                    )
            ),
            @ApiResponse(responseCode = "404", description = "未找到对应的种子文件")
    })
    @GetMapping("/download/{infoHash}")
    public ResponseEntity<?> downloadTorrent(
            @Parameter(description = "种子文件的唯一 infoHash", required = true, example = "207d69a047830000620c586e199fc3424e7e5dd8")
            @PathVariable String infoHash) {
        try {
            String url = torrentService.getTorrentUrl(infoHash);
            if (url != null) {
                return ResponseEntity.ok(Map.of("url", url));
            } else {
                return ResponseEntity.status(HttpStatus.NOT_FOUND).body("未找到对应的种子文件");
            }
        } catch (Exception ex) {
            return ResponseEntity.status(HttpStatus.INTERNAL_SERVER_ERROR).body("下载链接生成失败");
        }
    }

    @Operation(summary = "删除种子文件", description = "根据 infoHash 删除服务器上对应的 .torrent 文件和数据库记录")
    @ApiResponses({
            @ApiResponse(responseCode = "200", description = "删除成功"),
            @ApiResponse(responseCode = "404", description = "未找到对应的种子文件")
    })
    @DeleteMapping("/delete/{infoHash}")
    public ResponseEntity<?> deleteTorrent(
            @Parameter(description = "种子文件的唯一 infoHash", required = true, example = "207d69a047830000620c586e199fc3424e7e5dd8")
            @PathVariable String infoHash) {
        boolean deleted = torrentService.deleteTorrent(infoHash);
        if (deleted) {
            return ResponseEntity.ok("删除成功");
        } else {
            return ResponseEntity.status(HttpStatus.NOT_FOUND).body("未找到对应的种子文件");
        }
    }

    @Operation(summary = "获取种子列表", description = "查询所有已上传的种子记录，无分页")
    @ApiResponse(
            responseCode = "200",
            description = "返回种子列表",
            content = @Content(
                    mediaType = "application/json",
                    schema = @Schema(implementation = Torrent[].class),
                    examples = @ExampleObject(
                            name = "列表示例",
                            value = "[\n" +
                                    "  {\n" +
                                    "    \"id\": 1,\n" +
                                    "    \"infoHash\": \"ABCDEF1234567890ABCDEF1234567890ABCDEF12\",\n" +
                                    "    \"fileName\": \"movie.torrent\",\n" +
                                    "    \"userId\": 1,\n" +
                                    "    \"uploadTime\": \"2025-06-07T16:35:12\"\n" +
                                    "  },\n" +
                                    "  {\n" +
                                    "    \"id\": 2,\n" +
                                    "    \"infoHash\": \"123456ABCDEF123456ABCDEF123456ABCDEF1234\",\n" +
                                    "    \"fileName\": \"song.torrent\",\n" +
                                    "    \"userId\": 1,\n" +
                                    "    \"uploadTime\": \"2025-06-07T17:02:45\"\n" +
                                    "  }\n" +
                                    "]"
                    )
            )
    )
    @GetMapping("/getTorrentList")
    public ResponseEntity<?> getTorrentList() throws Exception {

        List<Torrent> list = torrentService.getTorrentList();
        return ResponseEntity.ok(list);

    }

}