// src/main/java/edu/bjtu/groupone/backend/api/user/UserController.java
package edu.bjtu.groupone.backend.api;

import edu.bjtu.groupone.backend.domain.dto.*;
import edu.bjtu.groupone.backend.domain.entity.User;
import edu.bjtu.groupone.backend.service.UserService;
import edu.bjtu.groupone.backend.utils.JwtUtils;
import edu.bjtu.groupone.backend.domain.entity.Result;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.enums.ParameterIn;
import io.swagger.v3.oas.annotations.media.Content;
import io.swagger.v3.oas.annotations.media.ExampleObject;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.tags.Tag;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

@CrossOrigin
@Tag(name = "用户相关接口") //输入http://localhost:8080/swagger-ui/index.html或http://localhost:8080/v3/api-docs/可查看接口文档
@RestController
@Slf4j
@RequestMapping("/api/user")
public class UserController {

    @Autowired
    private UserService userService;

    /**
     * 邮箱+密码 登录
     */
    @Operation(
            summary = "用户登录",
            description = "使用邮箱和密码进行登录，成功后返回 JWT token",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "登录成功",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "登录成功示例",
                                            value = "{\"code\":200,\"message\":\"操作成功\",\"data\":\"lyJhbGciOUzI1NiJ9.eyJpZCI6MSwidXNlcm5hbWUiOiJ0ZXN0IiwiZW1haWwiOiJ0ZXN0QGV4YW1wbGUuY29tIiwiaWF0IjoxNjU2NjU0MjQ0LCJleHAiOjE2NTY2NTc4NDR9.qBjQJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZw\"}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "邮箱或密码错误",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "登录失败示例",
                                            value = "{\"code\":400,\"message\":\"邮箱或密码错误\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/login")
    public Result<String> login(@Valid @RequestBody LoginRequest request) {
        User u = new User();
        u.setEmail(request.getEmail());
        u.setPassword(request.getPassword());
        User user = userService.login(u);
        if (user != null) {
            var payload = new ClaimsDTO();
            payload.setEmail(user.getEmail());
            payload.setUsername(user.getUsername());
            payload.setUserid(user.getUserId());
            String token = JwtUtils.generateJwt(payload);
            return Result.success(token);
        }
        return Result.error("邮箱或密码错误");
    }

    /**
     * 注册
     */
    @Operation(
            summary = "用户注册",
            description = "注册新用户，必须提供用户名、邮箱、验证码、密码和身份证号码",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "注册成功",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "注册成功示例",
                                            value = "{\"code\":200,\"message\":\"注册成功\",\"data\":null}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "参数错误或用户已存在",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "注册失败示例",
                                            value = "{\"code\":400,\"message\":\"邮箱已被注册\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/register")
    public Result<String> register(@Valid @RequestBody RegisterRequest request) {
        String email = request.getEmail();
        String code = request.getVerificationCode();
        String pwd = request.getPassword();

        // 验证验证码
        if (!userService.verifyCode(email, code)) {
            return Result.error("验证码错误或已过期");
        }

        // 检查邮箱是否已注册
        if (userService.isEmailExists(email)) {
            return Result.error("邮箱已被注册");
        }

        // 检查用户名是否已存在
        String username = request.getUsername();
        if (username != null && userService.isUsernameExists(username)) {
            return Result.error("用户名已被占用");
        }


        // 创建新用户
        User newUser = new User();
        newUser.setUsername(username);
        newUser.setEmail(email);
        newUser.setPassword(pwd);

        userService.register(newUser);
        return Result.success("注册成功");
    }

    /**
     * 发送注册验证码
     */
    @Operation(
            summary = "发送注册验证码",
            description = "向指定邮箱发送注册验证码",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "验证码已发送",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "发送成功示例",
                                            value = "{\"code\":200,\"message\":\"验证码已发送\",\"data\":null}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "邮箱不能为空",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "发送失败示例",
                                            value = "{\"code\":400,\"message\":\"邮箱不能为空\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/sendVerification")
    public Result<String> sendVerification(@RequestBody @Parameter(description = "邮箱请求体") EmailRequest request) {
        String email = request.getEmail();
        if (email == null) {
            return Result.error("邮箱不能为空");
        }
        userService.sendVerificationCode(email);
        return Result.success("验证码已发送");
    }

    @Operation(
            summary = "发送重置密码验证码",
            description = "向指定邮箱发送重置密码的验证码",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "重置验证码已发送",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "发送成功示例",
                                            value = "{\"code\":200,\"message\":\"重置验证码已发送\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/sendResetCode")
    public Result<String> sendResetCode(@RequestBody EmailRequest request) {
        String email = request.getEmail();
        userService.sendResetCode(email);
        return Result.success("重置验证码已发送");
    }

    @Operation(
            summary = "重置密码",
            description = "根据邮箱和验证码重置密码",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "密码已重置",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "重置成功示例",
                                            value = "{\"code\":200,\"message\":\"密码已重置\",\"data\":null}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "验证码错误或已过期",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "重置失败示例",
                                            value = "{\"code\":400,\"message\":\"验证码错误或已过期\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/resetPassword")
    public Result<String> resetPassword(@Valid @RequestBody ResetPasswordRequest request) {
        boolean ok = userService.resetPassword(request.getEmail(), request.getCode(), request.getNewPassword());
        return ok
                ? Result.success("密码已重置")
                : Result.error("验证码错误或已过期");
    }

    /**
     * 前端受保护页面获取当前用户信息
     */
    @Operation(
            summary = "获取当前用户信息",
            description = "需要提供有效的 JWT token",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "获取用户信息成功",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "获取成功示例",
                                            value = "{\"code\":200,\"message\":\"操作成功\",\"data\":{\"username\":\"test\",\"token\":\"eyJhbGciOiJIUzI1NiJ9.eyJpZCI6MSwidXNlcm5hbWUiOiJ0ZXN0IiwiZW1haWwiOiJ0ZXN0QGV4YW1wbGUuY29tIiwiaWF0IjoxNjU2NjU0MjQ0LCJleHAiOjE2NTY2NTc4NDR9.qBjQJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZ6ZJZw\"}}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "401",
                            description = "无效的 token",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "获取失败示例",
                                            value = "{\"code\":401,\"message\":\"无效的 token\",\"data\":null}"
                                    )
                            )
                    )
            }
    )

    @GetMapping("/me")
    public Result<UserInfoResponse> me(@Parameter(in = ParameterIn.HEADER, description = "JWT认证令牌") @RequestHeader("token") String token) {
        ClaimsDTO claims;
        try {
            claims = JwtUtils.parseJwt(token);  // 解析JWT
        } catch (RuntimeException e) {
            return Result.error("无效的 token: " + e.getMessage());  // 处理 JWT 异常
        }
        UserInfoResponse info = new UserInfoResponse(claims.getUsername(), claims.getUserid());
        return Result.success(info);
    }


    @DeleteMapping("/{userId}")
    @Operation(summary = "删除用户")
    public Result<String> deleteUser(@PathVariable int userId) {
        userService.deleteUser(userId);
        return Result.success("用户删除成功"); // 修改为中文
    }

    @PutMapping("/{userId}")
    @Operation(summary = "更新用户信息")
    public Result<String> updateUser(@PathVariable int userId, @RequestBody User user) {
        user.setUserId(userId);
        userService.updateUser(user);
        return Result.success("用户信息更新成功"); // 修改为中文
    }

    // 用户查询
    @GetMapping("/{userId}")
    @Operation(summary = "获取用户信息")
    public Result<User> getUser(@PathVariable int userId) {
        User user = userService.getUserById(userId);
        return Result.success(user);
    }

    // 所有用户查询
    @GetMapping("/all")
    @Operation(summary = "获取所有用户")
    public Result<List<User>> getAllUsers() {
        return Result.success(userService.getAllUsers());
    }

    /**
     * 刷新 Token
     */
    @Operation(
            summary = "刷新 JWT Token",
            description = "客户端调用此接口提供旧的 JWT（可能已过期），" +
                    "服务端会校验旧 token：如果仅仅是过期，则从中取出 Claims 并重新签发；" +
                    "如果签名无效或格式错误，则返回 401。",
            parameters = {
                    @Parameter(
                            in = ParameterIn.HEADER,
                            name = "token",
                            description = "旧的 JWT（在请求头中携带，即使已过期也可以）",
                            required = true,
                            schema = @Schema(type = "string", example = "eyJhbGciOiJIUzI1NiJ9.eyJpZCI6MSwidXNlcm5hbWUiOiJ1c2VyMSIsImVtYWlsIjoidXNlcjFAZXhhbXBsZS5jb20iLCJpYXQiOjE2NTAwMDAwMDAsImV4cCI6MTY1MDAwMTIwMH0.XYZxyzXYZxyzXYZxyzXYZxyzXYZxyzXYZxyz")
                    )
            },
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "刷新成功，返回新的 JWT",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "刷新成功示例",
                                            value = "{\"code\":200,\"message\":\"操作成功\",\"data\":\"eyJhbGciOiJIUzI1NiJ9.eyJpZCI6MSwidXNlcm5hbWUiOiJ1c2VyMSIsImVtYWlsIjoidXNlcjFAZXhhbXBsZS5jb20iLCJpYXQiOjE2NTAwMDAwMDAsImV4cCI6MTY1MDAwMjAwMH0.ABCabcABCabcABCabcABCabcABCabcABCabc\"}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "401",
                            description = "无效的 token（签名错误或格式不对）",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "签名错误示例",
                                            value = "{\"code\":401,\"message\":\"无效的 token\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @PostMapping("/refreshToken")
    public Result<String> refreshToken(
            @Parameter(in = ParameterIn.HEADER, description = "旧的 JWT 令牌", required = true)
            @RequestHeader("token") String oldToken
    ) {
        ClaimsDTO claims;
        try {
            // 尝试解析；如果尚未过期，parseJwt 会返回 Claims
            claims = JwtUtils.parseJwt(oldToken);
        } catch (ExpiredJwtException ex) {
            // 如果已过期，则从异常中获取过期前的 Claims
            claims = new ClaimsDTO(ex.getClaims());
        } catch (Exception ex) {
            // 其它解析错误（签名无效等）
            return Result.error("无效的 token");
        }

        // 从原有的 Claims 中提取用户信息，重新生成新的 token
        String newToken = JwtUtils.generateJwt(claims);
        return Result.success(newToken);
    }
    @Operation(
            summary = "查询用户剩余任务数量",
            description = "根据用户ID查询剩余未完成的任务数量（包括种子、帖子、评论）",
            responses = {
                    @ApiResponse(
                            responseCode = "200",
                            description = "查询成功",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "查询成功示例",
                                            value = "{\"code\":200,\"message\":\"操作成功\",\"data\":3}"
                                    )
                            )
                    ),
                    @ApiResponse(
                            responseCode = "400",
                            description = "用户不存在",
                            content = @Content(
                                    schema = @Schema(implementation = Result.class),
                                    examples = @ExampleObject(
                                            name = "查询失败示例",
                                            value = "{\"code\":400,\"message\":\"用户不存在\",\"data\":null}"
                                    )
                            )
                    )
            }
    )
    @GetMapping("/api/tasks/remaining/{userId}")
    public Result<Integer> getRemainingTasks(@PathVariable int userId) {
        int remainingTasks = userService.getRemainingTasks(userId);
        if (remainingTasks < 0) {
            return Result.error("用户不存在");
        }
        return Result.success(remainingTasks);
    }
}