package edu.bjtu.groupone.backend;

import edu.bjtu.groupone.backend.api.TorrentController;
import edu.bjtu.groupone.backend.domain.entity.Torrent;
import edu.bjtu.groupone.backend.service.TorrentService;
import edu.bjtu.groupone.backend.utils.GetTokenUserId;
import jakarta.servlet.http.HttpServletRequest;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.junit.jupiter.MockitoExtension;
import org.springframework.core.io.ByteArrayResource;
import org.springframework.core.io.Resource;
import org.springframework.http.ResponseEntity;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.web.multipart.MultipartFile;

@ExtendWith({MockitoExtension.class})
public class TorrentControllerTest {
    @Mock
    private TorrentService torrentService;
    @InjectMocks
    private TorrentController torrentController;
    @Mock
    private HttpServletRequest request;

    public TorrentControllerTest() {
    }

    @Test
    public void uploadTorrent_shouldReturnTorrent_whenUserIdValid() throws Exception {
        MultipartFile file = new MockMultipartFile("file", "test.torrent", "application/x-bittorrent", "dummy data".getBytes());
        MockedStatic<GetTokenUserId> utilities = Mockito.mockStatic(GetTokenUserId.class);

        try {
            utilities.when(() -> {
                GetTokenUserId.getUserId(this.request);
            }).thenReturn("123");
            Torrent expectedTorrent = new Torrent();
            expectedTorrent.setId(1L);
            expectedTorrent.setName("testfile");
            expectedTorrent.setInfoHash("fakehash");
            expectedTorrent.setSize(12345L);
            expectedTorrent.setUploaderId(123L);
            Mockito.when(this.torrentService.uploadTorrent(file, 123L)).thenReturn(expectedTorrent);
            ResponseEntity<?> response = this.torrentController.uploadTorrent(file, this.request);
            Assertions.assertThat(response.getStatusCodeValue()).isEqualTo(200);
            Assertions.assertThat(response.getBody()).isEqualTo(expectedTorrent);
            ((TorrentService)Mockito.verify(this.torrentService, Mockito.times(1))).uploadTorrent(file, 123L);
        } catch (Throwable var6) {
            if (utilities != null) {
                try {
                    utilities.close();
                } catch (Throwable var5) {
                    var6.addSuppressed(var5);
                }
            }

            throw var6;
        }

        if (utilities != null) {
            utilities.close();
        }

    }

    @Test
    public void uploadTorrent_shouldReturnUnauthorized_whenUserIdNull() throws Exception {
        MockedStatic<GetTokenUserId> utilities = Mockito.mockStatic(GetTokenUserId.class);

        try {
            utilities.when(() -> {
                GetTokenUserId.getUserId(this.request);
            }).thenReturn((Object)null);
            MultipartFile file = new MockMultipartFile("file", "test.torrent", "application/x-bittorrent", "dummy data".getBytes());
            ResponseEntity<?> response = this.torrentController.uploadTorrent(file, this.request);
            Assertions.assertThat(response.getStatusCodeValue()).isEqualTo(401);
            Assertions.assertThat(response.getBody()).isEqualTo("Token无效或缺失");
            Mockito.verifyNoInteractions(new Object[]{this.torrentService});
        } catch (Throwable var5) {
            if (utilities != null) {
                try {
                    utilities.close();
                } catch (Throwable var4) {
                    var5.addSuppressed(var4);
                }
            }

            throw var5;
        }

        if (utilities != null) {
            utilities.close();
        }

    }

    @Test
    public void downloadTorrent_shouldReturnResource_whenFound() throws Exception {
        String infoHash = "fakehash";
        byte[] data = "torrent data".getBytes();
        Resource resource = new ByteArrayResource(data);
        Mockito.when(this.torrentService.downloadTorrent(infoHash)).thenReturn(resource);
        ResponseEntity<Resource> response = this.torrentController.downloadTorrent(infoHash);
        Assertions.assertThat(response.getStatusCodeValue()).isEqualTo(200);
        Assertions.assertThat(response.getHeaders().getFirst("Content-Disposition")).isEqualTo("attachment; filename=\"" + infoHash + ".torrent\"");
        Assertions.assertThat((Resource)response.getBody()).isEqualTo(resource);
        ((TorrentService)Mockito.verify(this.torrentService, Mockito.times(1))).downloadTorrent(infoHash);
    }

    @Test
    public void downloadTorrent_shouldReturnNotFound_whenException() throws Exception {
        String infoHash = "notexist";
        Mockito.when(this.torrentService.downloadTorrent(infoHash)).thenThrow(new Throwable[]{new RuntimeException("Not found")});
        ResponseEntity<Resource> response = this.torrentController.downloadTorrent(infoHash);
        Assertions.assertThat(response.getStatusCodeValue()).isEqualTo(404);
        Assertions.assertThat((Resource)response.getBody()).isNull();
        ((TorrentService)Mockito.verify(this.torrentService, Mockito.times(1))).downloadTorrent(infoHash);
    }
}