// src/store/userSlice.ts
import { createSlice, createAsyncThunk, type PayloadAction } from '@reduxjs/toolkit';
import type { UserInfo } from '../../api/User/type';
import UserAPi from '../../api/User/UserApi';

// 定义用户信息的类型
interface UserState {
    username: string;
    userid: string;
    email: string;
    status: 'idle' | 'loading' | 'succeeded' | 'failed';
    error: string | null;
}

// 定义初始状态
const initialState: UserState = {
    username: '',
    userid: '',
    email: '',
    status: 'idle',
    error: null,
};


// 创建异步 action，用于获取用户信息
export const getUserInfo = createAsyncThunk<
    UserInfo,
    void,
    {rejectValue:string}
>(
    'user/getUserInfo',
    async (_, { rejectWithValue }) => {
        const response = await UserAPi.getMe();
        if (response.data.code == 0) {
            return response.data.data;
        } else {
            return rejectWithValue(response.data.message);
        }
    } 
);

// 创建 userSlice
const userSlice = createSlice({
    name: 'user',
    initialState,
    reducers: {
        // 可以在这里处理同步操作，如修改用户名等
        setUser: (state, action: PayloadAction<string>) => {
            state.username = action.payload;
        },
    },
    extraReducers: (builder) => {
        builder
            .addCase(getUserInfo.pending, (state) => {
                state.status = 'loading';
            })
            .addCase(getUserInfo.fulfilled, (state, action: PayloadAction<UserInfo>) => {
                state.status = 'succeeded';
                state.username = action.payload.username;
                state.userid = action.payload.userid;
            })
            .addCase(getUserInfo.rejected, (state, action) => {
                state.status = 'failed';
                state.error = action.error.message ?? 'Unknown error';
            });
    },
});

// 导出 actions 和 reducer
export const { setUser } = userSlice.actions;
export default userSlice.reducer;
