import type { UploadFile } from 'antd/es/upload';

// ==================== 基础类型 ====================

export interface User {
    userId: string;
    username: string;
}

export interface HistoryUser extends User {
    uploadTotal: string;
}

// ==================== 评论相关 ====================

export interface Comment {
    id?: string;
    content: string;
    author: string;
    authorId?: string;
    createdAt?: string;
    child: Comment[];
}

// ==================== 版本管理相关 ====================

// 用于表单创建的版本信息
export interface VersionFormData {
    version: string;
    versionDescription: string;
    seedFile?: UploadFile; // 表单中的文件对象
}

// 用于展示的版本信息（后端返回）
export interface Version {
    version: string;
    versionDescription: string;
    seedFile: string; // 文件URL或路径
    uploadedAt?: string;
    fileSize?: number;
}

// ==================== 作品相关 ====================

// 基础信息（表单用）
export interface BasicInfo {
    artworkName: string;
    artworkCategory: ArtworkCategory;
    artworkDescription: string;
}

// 封面信息（表单用）
export interface CoverInfo {
    coverFile?: UploadFile;
}

// 表单数据结构
export interface WorkFormData {
    basicInfo: BasicInfo;
    coverInfo: CoverInfo;
    versions: VersionFormData[];
}

// 完整的作品数据（后端返回/展示用）
export interface ArtworkData {
    artworkId: string;
    artworkCover: string;
    author: string;
    authorId: string;
    artworkName: string;
    artworkCategory: ArtworkCategory;
    artworkDescription: string;
    versionList: Version[];
    comments: Comment[];
    usersSeedingCurrently: User[];
    usersSeedingHistory: HistoryUser[];
    createdAt?: string;
    updatedAt?: string;
    downloadCount?: number;
    likeCount?: number;
}

// ==================== 表单相关 ====================

export interface StepFormProps {
    data: WorkFormData;
    onUpdate: (field: keyof WorkFormData, value: BasicInfo | CoverInfo | VersionFormData[]) => void;
    onNext?: () => void;
    onPrev?: () => void;
}

// ==================== 常量定义 ====================

export const ARTWORK_CATEGORIES = [
    { label: '概念设计', value: '概念设计' },
    { label: 'UI/UX设计', value: 'UI/UX设计' },
    { label: '3D建模', value: '3D建模' },
    { label: '摄影作品', value: '摄影作品' },
    { label: '音乐制作', value: '音乐制作' },
    { label: '前端开发', value: '前端开发' },
    { label: '插画艺术', value: '插画艺术' },
    { label: '动画制作', value: '动画制作' },
    { label: '视频剪辑', value: '视频剪辑' },
    { label: '其他', value: '其他' },
];

// 提取分类值的联合类型
export type ArtworkCategory = '概念设计' | 'UI/UX设计' | '3D建模' | '摄影作品' | '音乐制作' | '前端开发' | '插画艺术' | '动画制作' | '视频剪辑' | '其他';

// 分类选项类型（用于 Select 组件）
export interface ArtworkCategoryOption {
    label: string;
    value: ArtworkCategory;
}

// ==================== 工具函数 ====================

/**
 * 解析上传大小字符串为MB数值
 * @param uploadStr 上传大小字符串，如 "1.5GB", "500MB"
 * @returns 大小值（以MB为单位）
 */
export const parseUploadSize = (uploadStr: string): number => {
    const match = uploadStr.match(/(\d+(?:\.\d+)?)\s*(GB|MB|KB|TB)?/i);
    if (!match) return 0;
    
    const size = parseFloat(match[1]);
    const unit = (match[2] || '').toUpperCase();
    
    const multipliers: Record<string, number> = {
        'TB': 1024 * 1024,
        'GB': 1024,
        'MB': 1,
        'KB': 1 / 1024,
        '': 1
    };
    
    return size * (multipliers[unit] || 1);
};

/**
 * 格式化文件大小为可读字符串
 * @param sizeInMB 文件大小（MB）
 * @returns 格式化后的字符串
 */
export const formatFileSize = (sizeInMB: number): string => {
    if (sizeInMB >= 1024) {
        return `${(sizeInMB / 1024).toFixed(1)}GB`;
    } else if (sizeInMB >= 1) {
        return `${sizeInMB.toFixed(1)}MB`;
    } else {
        return `${(sizeInMB * 1024).toFixed(0)}KB`;
    }
};

/**
 * 格式化日期
 * @param dateString 日期字符串
 * @returns 格式化后的日期字符串
 */
export const formatDate = (dateString?: string): string => {
    if (!dateString) return '';
    
    const date = new Date(dateString);
    const now = new Date();
    const diffInHours = (now.getTime() - date.getTime()) / (1000 * 60 * 60);
    
    if (diffInHours < 1) {
        return '刚刚';
    } else if (diffInHours < 24) {
        return `${Math.floor(diffInHours)}小时前`;
    } else if (diffInHours < 24 * 7) {
        return `${Math.floor(diffInHours / 24)}天前`;
    } else {
        return date.toLocaleDateString('zh-CN');
    }
};

/**
 * 验证文件大小是否符合要求
 * @param fileSize 文件大小（字节）
 * @param maxSizeMB 最大允许大小（MB）
 * @returns 是否符合要求
 */
export const isValidFileSize = (fileSize: number, maxSizeMB: number): boolean => {
    const fileSizeMB = fileSize / (1024 * 1024);
    return fileSizeMB <= maxSizeMB;
};

/**
 * 验证文件类型是否符合要求
 * @param fileName 文件名
 * @param allowedExtensions 允许的扩展名数组
 * @returns 是否符合要求
 */
export const isValidFileType = (fileName: string, allowedExtensions: string[]): boolean => {
    if (!fileName) return false;
    const extension = fileName.toLowerCase().split('.').pop();
    return extension ? allowedExtensions.includes(extension) : false;
};

/**
 * 验证图片文件类型
 * @param fileName 文件名
 * @returns 是否为有效图片
 */
export const isValidImageFile = (fileName: string): boolean => {
    return isValidFileType(fileName, ['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg']);
};

/**
 * 安全地解析JSON字符串
 * @param jsonString JSON字符串
 * @returns 解析结果或null
 */
export const safeJsonParse = <T = unknown>(jsonString: string): T | null => {
    try {
        return JSON.parse(jsonString) as T;
    } catch {
        return null;
    }
};

// ==================== 类型守卫 ====================

export const isValidArtworkCategory = (category: string): category is ArtworkCategory => {
    return ARTWORK_CATEGORIES.some(cat => cat.value === category);
};

export const isComment = (obj: unknown): obj is Comment => {
    return typeof obj === 'object' && 
           obj !== null && 
           'content' in obj && 
           'author' in obj && 
           'child' in obj &&
           typeof (obj as Record<string, unknown>).content === 'string' && 
           typeof (obj as Record<string, unknown>).author === 'string' && 
           Array.isArray((obj as Record<string, unknown>).child);
};

export const isUser = (obj: unknown): obj is User => {
    return typeof obj === 'object' && 
           obj !== null && 
           'userId' in obj && 
           'username' in obj &&
           typeof (obj as Record<string, unknown>).userId === 'string' && 
           typeof (obj as Record<string, unknown>).username === 'string';
};

export const isHistoryUser = (obj: unknown): obj is HistoryUser => {
    return isUser(obj) && 
           'uploadTotal' in obj &&
           typeof (obj as Record<string, unknown>).uploadTotal === 'string';
};

export const isVersion = (obj: unknown): obj is Version => {
    return typeof obj === 'object' && 
           obj !== null && 
           'version' in obj && 
           'versionDescription' in obj && 
           'seedFile' in obj &&
           typeof (obj as Record<string, unknown>).version === 'string' && 
           typeof (obj as Record<string, unknown>).versionDescription === 'string' && 
           typeof (obj as Record<string, unknown>).seedFile === 'string';
};

export const isArtworkData = (obj: unknown): obj is ArtworkData => {
    if (typeof obj !== 'object' || obj === null) return false;
    
    const artwork = obj as Record<string, unknown>;
    
    return typeof artwork.artworkId === 'string' &&
           typeof artwork.artworkCover === 'string' &&
           typeof artwork.author === 'string' &&
           typeof artwork.authorId === 'string' &&
           typeof artwork.artworkName === 'string' &&
           typeof artwork.artworkCategory === 'string' &&
           typeof artwork.artworkDescription === 'string' &&
           Array.isArray(artwork.versionList) &&
           Array.isArray(artwork.comments) &&
           Array.isArray(artwork.usersSeedingCurrently) &&
           Array.isArray(artwork.usersSeedingHistory);
};