import type { ArtworkData, Comment } from './types';

// 生成随机时间的工具函数
const generateRandomDate = (daysAgo: number): string => {
    const date = new Date();
    date.setDate(date.getDate() - Math.floor(Math.random() * daysAgo));
    return date.toLocaleString('zh-CN', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit'
    });
};

// 通用评论数据
const baseComments: Comment[] = [
    {
        id: 'comment_1',
        content: '这个作品真的很棒！设计思路很新颖，学到了很多东西。',
        author: '设计爱好者小王',
        authorId: 'user_wang_001',
        createdAt: generateRandomDate(3),
        child: [
            {
                id: 'comment_1_1',
                content: '同感！特别是色彩搭配的部分，很有启发性。',
                author: '视觉设计师李雷',
                authorId: 'user_lilei_002',
                createdAt: generateRandomDate(2),
                child: [],
            },
            {
                id: 'comment_1_2',
                content: '确实，作者的审美很在线，期待更多作品！',
                author: '创意总监韩梅梅',
                authorId: 'user_hanmeimei_003',
                createdAt: generateRandomDate(2),
                child: [
                    {
                        id: 'comment_1_2_1',
                        content: '@创意总监韩梅梅 您说得对，这种风格确实很难得',
                        author: '设计爱好者小王',
                        authorId: 'user_wang_001',
                        createdAt: generateRandomDate(1),
                        child: [],
                    }
                ],
            },
        ],
    },
    {
        id: 'comment_2',
        content: '技术实现方面有什么特别的难点吗？想了解一下制作过程。',
        author: '技术控阿明',
        authorId: 'user_aming_004',
        createdAt: generateRandomDate(5),
        child: [],
    },
    {
        id: 'comment_3',
        content: '看起来很专业，请问这是用什么软件制作的？',
        author: '新手小白',
        authorId: 'user_xiaobai_005',
        createdAt: generateRandomDate(4),
        child: [
            {
                id: 'comment_3_1',
                content: '从效果来看应该是用Adobe全家桶，主要是PS和AI',
                author: '软件专家老刘',
                authorId: 'user_laoliu_006',
                createdAt: generateRandomDate(3),
                child: [],
            }
        ],
    },
    {
        id: 'comment_4',
        content: '色彩运用得很巧妙，给人一种很舒服的视觉感受。',
        author: '色彩搭配师小张',
        authorId: 'user_xiaozhang_007',
        createdAt: generateRandomDate(6),
        child: [],
    },
    {
        id: 'comment_5',
        content: '作品质量很高，但是感觉在某些细节上还可以再优化一下。',
        author: '完美主义者',
        authorId: 'user_perfect_008',
        createdAt: generateRandomDate(7),
        child: [
            {
                id: 'comment_5_1',
                content: '能具体说说哪些地方可以优化吗？我也想学习学习',
                author: '求知者小陈',
                authorId: 'user_xiaochen_009',
                createdAt: generateRandomDate(6),
                child: [],
            }
        ],
    },
    {
        id: 'comment_6',
        content: '已经下载收藏了，感谢分享！🎉',
        author: '收藏家大佬',
        authorId: 'user_collector_010',
        createdAt: generateRandomDate(8),
        child: [],
    },
    {
        id: 'comment_7',
        content: '这个风格很适合我正在做的项目，能不能商用呢？',
        author: '商务合作方',
        authorId: 'user_business_011',
        createdAt: generateRandomDate(9),
        child: [],
    },
    {
        id: 'comment_8',
        content: '教程什么时候出？期待大神的分享！',
        author: '学习狂人',
        authorId: 'user_learner_012',
        createdAt: generateRandomDate(10),
        child: [],
    }
];

// 作品数据集合
export const mockArtworks: Record<string, ArtworkData> = {
    '1': {
        artworkId: '1',
        artworkCover: 'https://picsum.photos/300/400?random=1',
        author: '视觉设计师张三',
        authorId: 'author_zhangsan_001',
        artworkName: '未来城市概念设计',
        artworkCategory: '概念设计',
        comments: [],
        artworkDescription: `# 未来城市概念设计

这是一个关于2050年智慧城市的概念设计作品。整个设计融合了**可持续发展**、**人工智能**和**绿色科技**的理念。

## 设计理念

- 🌱 **生态友好**：建筑与自然和谐共生
- 🤖 **智能化**：AI驱动的城市管理系统
- 🚗 **零排放**：全面电动化的交通系统
- 🏗️ **模块化**：可扩展的建筑结构

## 技术特点

采用了最新的**参数化设计**方法，通过算法生成建筑形态，确保每个结构都能最大化利用自然光和风能。

*希望这个设计能为未来城市规划提供一些启发。*`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:future_city_v1_0&dn=未来城市v1.0.zip',
                versionDescription: `## 初始版本 v1.0

### 包含内容
- 🎨 **高分辨率效果图** (4K)
- 📐 **CAD源文件** (.dwg)
- 🖼️ **PSD分层文件**
- 📄 **设计说明文档**

### 更新说明
首次发布，包含完整的设计方案和素材文件。`,
            },
            {
                version: '1.1',
                seedFile: 'magnet:?xt=urn:btih:future_city_v1_1&dn=未来城市v1.1.zip',
                versionDescription: `## 更新版本 v1.1

### 新增内容
- 🌃 **夜景渲染图**
- 🎬 **动画演示视频** (1080p)
- 🔧 **Blender源文件**

### 修复内容
- 修复了部分贴图丢失问题
- 优化了文件结构
- 添加了英文版说明文档`,
            },
        ],
        usersSeedingCurrently: [
            { username: '设计师小李', userId: 'user_xiaoli_101' },
            { username: '建筑师王工', userId: 'user_wanggong_102' },
            { username: '学生小赵', userId: 'user_xiaozhao_103' },
            { username: '创意总监', userId: 'user_director_104' },
        ],
        usersSeedingHistory: [
            { username: '资深下载者', uploadTotal: '156.8GB', userId: 'hist_user_001' },
            { username: '设计素材库', uploadTotal: '89.2GB', userId: 'hist_user_002' },
            { username: '创意工作室', uploadTotal: '67.5GB', userId: 'hist_user_003' },
            { username: '学院资源组', uploadTotal: '45.3GB', userId: 'hist_user_004' },
            { username: '独立设计师', uploadTotal: '23.7GB', userId: 'hist_user_005' },
        ],
    },
    
    '2': {
        artworkId: '2',
        artworkCover: 'https://picsum.photos/300/400?random=2',
        author: 'UI设计师李四',
        authorId: 'author_lisi_002',
        artworkName: '移动应用界面设计套件',
        artworkCategory: 'UI/UX设计',
        comments: [],
        artworkDescription: `# 移动应用界面设计套件

一套完整的移动端UI设计规范和组件库，包含**100+个精美界面**和**500+个设计组件**。

## 包含内容

### 📱 界面设计
- 登录注册页面
- 主页和导航
- 商品展示页面
- 个人中心
- 设置页面

### 🎨 设计系统
- **颜色规范**：主色调、辅助色、状态色
- **字体系统**：标题、正文、注释文字
- **图标库**：线性图标、填充图标
- **组件库**：按钮、输入框、卡片等`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:ui_kit_v1_0&dn=UI设计套件v1.0.zip',
                versionDescription: `## 基础版本 v1.0

### 核心功能
- 📱 **50个界面模板**
- 🎨 **基础组件库**
- 📋 **设计规范文档**
- 🎯 **Sketch源文件**`,
            },
        ],
        usersSeedingCurrently: [
            { username: 'UI设计新手', userId: 'user_ui_newbie_201' },
            { username: '产品经理小王', userId: 'user_pm_wang_202' },
        ],
        usersSeedingHistory: [
            { username: 'UI设计公司', uploadTotal: '78.9GB', userId: 'hist_ui_001' },
            { username: '设计师联盟', uploadTotal: '45.6GB', userId: 'hist_ui_002' },
            { username: '学习小组', uploadTotal: '23.4GB', userId: 'hist_ui_003' },
        ],
    },

    '3': {
        artworkId: '3',
        artworkCover: 'https://picsum.photos/300/400?random=6',
        author: '刘松林',
        authorId: '2',
        artworkName: 'React组件库开发指南',
        artworkCategory: '前端开发',
        comments: [],
        artworkDescription: `# React组件库开发指南

一套完整的**企业级React组件库**开发教程和源码，包含从设计到发布的完整流程。

## 项目特色

### 🚀 技术栈
- **React 18** + **TypeScript**
- **Styled-components** 样式解决方案
- **Storybook** 组件文档
- **Jest** + **Testing Library** 测试

### 📦 组件覆盖
- **基础组件**: Button, Input, Select, Modal等
- **布局组件**: Grid, Layout, Container等  
- **数据展示**: Table, List, Card, Timeline等`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:react_components_v1_0&dn=React组件库v1.0.zip',
                versionDescription: `## 基础版本 v1.0

### 核心内容
- 📚 **完整教程文档** (50+页)
- 💻 **基础组件源码** (20+个组件)
- 🧪 **单元测试示例**
- 📖 **Storybook配置**`,
            },
        ],
        usersSeedingCurrently: [
            { username: '前端新手小李', userId: 'user_frontend_newbie_601' },
            { username: 'React爱好者', userId: 'user_react_fan_602' },
        ],
        usersSeedingHistory: [
            { username: '大厂前端团队', uploadTotal: '567.8GB', userId: 'hist_dev_001' },
            { username: '开源社区', uploadTotal: '234.5GB', userId: 'hist_dev_002' },
            { username: '技术培训机构', uploadTotal: '189.7GB', userId: 'hist_dev_003' },
        ],
    },

    '4': {
        artworkId: '4',
        artworkCover: 'https://picsum.photos/300/400?random=7',
        author: '3D艺术家王五',
        authorId: 'author_wangwu_004',
        artworkName: '机械战士3D模型',
        artworkCategory: '3D建模',
        comments: [],
        artworkDescription: `# 机械战士3D模型

一个高精度的科幻机械战士3D模型，包含完整的材质贴图和动画骨骼系统。

## 模型特点

### 🤖 设计风格
- 未来科幻风格
- 硬表面建模技术
- PBR材质工作流
- 模块化装备系统

### 📐 技术规格
- **面数**: 25,000 三角面
- **贴图分辨率**: 4K PBR贴图组
- **骨骼系统**: 完整人形骨骼
- **动画**: 10个基础动作

适用于游戏开发、影视制作和3D打印。`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:mech_warrior_v1_0&dn=机械战士v1.0.zip',
                versionDescription: `## 基础版本 v1.0

### 核心内容
- 🎯 **高精度3D模型** (.fbx, .obj)
- 🎨 **4K PBR贴图组**
- 🦴 **完整骨骼系统**
- 🎬 **基础动画文件**`,
            },
        ],
        usersSeedingCurrently: [
            { username: '游戏开发者小陈', userId: 'user_gamedev_301' },
            { username: '3D建模师', userId: 'user_3dmodeler_302' },
        ],
        usersSeedingHistory: [
            { username: '游戏工作室', uploadTotal: '234.5GB', userId: 'hist_3d_001' },
            { username: '影视特效团队', uploadTotal: '178.3GB', userId: 'hist_3d_002' },
            { username: '独立开发者', uploadTotal: '98.7GB', userId: 'hist_3d_003' },
        ],
    },

    '5': {
        artworkId: '5',
        artworkCover: 'https://picsum.photos/300/400?random=8',
        author: '摄影师赵六',
        authorId: 'author_zhaoliu_005',
        artworkName: '城市夜景摄影集',
        artworkCategory: '摄影作品',
        comments: [],
        artworkDescription: `# 城市夜景摄影集

一组精美的城市夜景摄影作品，捕捉了都市夜晚的璀璨光影。

## 作品特色

### 📸 拍摄技法
- 长曝光技术
- HDR合成处理
- 光轨艺术表现
- 城市建筑几何美学

### 🎨 后期处理
- **RAW格式**: 无损原始文件
- **精修版本**: Lightroom + Photoshop
- **色彩分级**: 电影级调色
- **分辨率**: 6000x4000像素

包含20张高分辨率摄影作品，适合商业使用和艺术收藏。`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:night_city_photos_v1_0&dn=城市夜景v1.0.zip',
                versionDescription: `## 完整版本 v1.0

### 包含内容
- 📷 **RAW原始文件** (20张)
- 🎨 **精修JPG版本** (高分辨率)
- 📋 **拍摄参数记录**
- 📍 **拍摄地点信息**`,
            },
        ],
        usersSeedingCurrently: [
            { username: '摄影爱好者小林', userId: 'user_photo_401' },
            { username: '设计师小美', userId: 'user_designer_402' },
        ],
        usersSeedingHistory: [
            { username: '摄影工作室', uploadTotal: '445.8GB', userId: 'hist_photo_001' },
            { username: '商业摄影师', uploadTotal: '367.2GB', userId: 'hist_photo_002' },
            { username: '摄影学院', uploadTotal: '289.1GB', userId: 'hist_photo_003' },
        ],
    },

    '6': {
        artworkId: '6',
        artworkCover: 'https://picsum.photos/300/400?random=9',
        author: '插画师孙七',
        authorId: 'author_sunqi_006',
        artworkName: '奇幻世界插画系列',
        artworkCategory: '插画艺术',
        comments: [],
        artworkDescription: `# 奇幻世界插画系列

一套充满想象力的奇幻题材插画作品，包含角色设计、场景概念图和完整插图。

## 创作风格

### 🎨 艺术特色
- 欧美奇幻风格
- 数字绘画技法
- 丰富色彩层次
- 细腻光影表现

### 📚 作品内容
- **角色设计**: 15个原创角色
- **场景概念**: 8个奇幻场景
- **完整插图**: 12张精美插画
- **线稿资源**: 黑白线稿版本

适合游戏美术、小说封面、卡牌设计等多种用途。`,
        versionList: [
            {
                version: '1.0',
                seedFile: 'magnet:?xt=urn:btih:fantasy_art_v1_0&dn=奇幻插画v1.0.zip',
                versionDescription: `## 标准版本 v1.0

### 核心内容
- 🎨 **高分辨率插画** (35张)
- ✏️ **线稿资源包**
- 🎯 **PSD分层文件**
- 📖 **创作过程记录**`,
            },
        ],
        usersSeedingCurrently: [
            { username: '插画学习者', userId: 'user_illustrator_501' },
            { username: '游戏美术师', userId: 'user_gameart_502' },
        ],
        usersSeedingHistory: [
            { username: '插画师联盟', uploadTotal: '378.9GB', userId: 'hist_art_001' },
            { username: '游戏美术团队', uploadTotal: '256.4GB', userId: 'hist_art_002' },
            { username: '艺术学院', uploadTotal: '189.6GB', userId: 'hist_art_003' },
        ],
    },
};

// 获取指定作品的评论数据
export const getCommentsForArtwork = (artworkId: string): Comment[] => {
    const commentVariations: Record<string, Comment[]> = {
        '1': baseComments,
        '2': baseComments.slice(0, 5).map(comment => ({
            ...comment,
            id: `ui_${comment.id}`,
            content: comment.content.replace('作品', 'UI套件').replace('设计', '界面设计'),
        })),
        '3': [
            {
                id: 'dev_comment_1',
                content: '这个组件库的设计思路很棒！TypeScript类型定义特别完善。',
                author: '前端架构师张工',
                authorId: 'user_architect_zhang',
                createdAt: generateRandomDate(2),
                child: [
                    {
                        id: 'dev_comment_1_1',
                        content: '同感！特别是组件API的设计，很符合React的设计理念。',
                        author: 'React核心开发者',
                        authorId: 'user_react_core',
                        createdAt: generateRandomDate(1),
                        child: [],
                    }
                ],
            },
            {
                id: 'dev_comment_2',
                content: '构建配置写得很详细，我们团队已经参考这个搭建了自己的组件库。',
                author: '技术负责人小刘',
                authorId: 'user_tech_liu',
                createdAt: generateRandomDate(3),
                child: [],
            },
        ],
        '4': baseComments.slice(0, 4).map(comment => ({
            ...comment,
            id: `3d_${comment.id}`,
            content: comment.content.replace('作品', '3D模型').replace('设计', '建模'),
        })),
        '5': baseComments.slice(0, 6).map(comment => ({
            ...comment,
            id: `photo_${comment.id}`,
            content: comment.content.replace('作品', '摄影作品').replace('设计思路', '拍摄技法'),
        })),
        '6': baseComments.slice(0, 5).map(comment => ({
            ...comment,
            id: `art_${comment.id}`,
            content: comment.content.replace('作品', '插画').replace('设计', '绘画'),
        })),
    };
    
    return commentVariations[artworkId] || baseComments;
};

// 根据作品ID获取作品数据
export const getArtworkById = (artworkId: string): ArtworkData | null => {
    const artwork = mockArtworks[artworkId];
    if (artwork) {
        return {
            ...artwork,
            comments: getCommentsForArtwork(artworkId)
        };
    }
    return null;
};

// 获取所有作品列表
export const getAllArtworks = (): ArtworkData[] => {
    return Object.values(mockArtworks);
};

// 按分类获取作品
export const getArtworksByCategory = (category: string): ArtworkData[] => {
    return Object.values(mockArtworks).filter(artwork => artwork.artworkCategory === category);
};

// 搜索作品
export const searchArtworks = (keyword: string): ArtworkData[] => {
    const lowerKeyword = keyword.toLowerCase();
    return Object.values(mockArtworks).filter(artwork => 
        artwork.artworkName.toLowerCase().includes(lowerKeyword) ||
        artwork.artworkDescription.toLowerCase().includes(lowerKeyword) ||
        artwork.author.toLowerCase().includes(lowerKeyword) ||
        artwork.artworkCategory.toLowerCase().includes(lowerKeyword)
    );
};