// src/api/__tests__/comment.test.js
const axios = require('axios');
jest.mock('axios');

const {
    createComment,
    deleteComment,
    updateComment,
    getCommentsByPostId,
    likeComment,
    unlikeComment
} = require('../comment');

describe('Comment API Tests', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });

    test('createComment should send POST request', async () => {
        const mockData = { userid: 1, postid: 2, postCommentcontent: 'Test comment' };
        axios.post.mockResolvedValue({ data: mockData });

        const response = await createComment(mockData);

        expect(axios.post).toHaveBeenCalledWith('http://localhost:8080/comment/create', mockData);
        expect(response.data).toEqual(mockData);
    });

    test('deleteComment should send DELETE request', async () => {
        axios.delete.mockResolvedValue({ data: true });

        const response = await deleteComment(123);

        expect(axios.delete).toHaveBeenCalledWith('http://localhost:8080/comment/delete/123');
        expect(response.data).toBe(true);
    });

    test('updateComment should send PUT request', async () => {
        const updatedData = { commentid: 1, postCommentcontent: 'Updated comment' };
        axios.put.mockResolvedValue({ data: true });

        const response = await updateComment(updatedData);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/comment/update', updatedData);
        expect(response.data).toBe(true);
    });

    test('getCommentsByPostId should fetch comments by post ID', async () => {
        const mockResponse = { data: [{ commentid: 1, postCommentcontent: 'Nice!' }] };
        axios.get.mockResolvedValue(mockResponse);

        const response = await getCommentsByPostId(5);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/comment/post/5');
        expect(response).toEqual(mockResponse.data);
    });

    test('getCommentsByPostId should return empty array on error', async () => {
        axios.get.mockRejectedValue(new Error('Network Error'));

        const response = await getCommentsByPostId(99);

        expect(response).toEqual([]);
    });

    test('likeComment should send POST request', async () => {
        axios.post.mockResolvedValue({ data: true });

        const response = await likeComment(10);

        expect(axios.post).toHaveBeenCalledWith('http://localhost:8080/comment/like/10');
        expect(response.data).toBe(true);
    });

    test('unlikeComment should send POST request', async () => {
        axios.post.mockResolvedValue({ data: true });

        const response = await unlikeComment(10);

        expect(axios.post).toHaveBeenCalledWith('http://localhost:8080/comment/unlike/10');
        expect(response.data).toBe(true);
    });
});
