const axios = require('axios');
const MockAdapter = require('axios-mock-adapter');

const {
    createPost,
    togglePinPost,
    deletePost,
    updatePost,
    searchPosts,
    likePost,
    unlikePost,
    pinPost,
    unpinPost,
    findPostsByUserId,
    findPinnedPosts,
    getAllPostsSorted,
    getPostById
} = require('../post'); // 注意根据你的实际路径修改

jest.mock('axios');

describe('Post API Tests', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });

    test('createPost should post form data', async () => {
        const formData = new FormData();
        formData.append('userid', '1');
        formData.append('post_title', 'Test');
        axios.post.mockResolvedValue({ data: true });

        const response = await createPost(formData);

        expect(axios.post).toHaveBeenCalledWith(
            'http://localhost:8080/post/create',
            formData,
            { headers: { 'Content-Type': 'multipart/form-data' } }
        );
        expect(response.data).toBe(true);
    });

    test('togglePinPost should send PUT request', async () => {
        axios.put.mockResolvedValue({ data: true });

        const response = await togglePinPost(123);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/post/togglePin/123');
        expect(response.data).toBe(true);
    });

    test('deletePost should send DELETE request', async () => {
        axios.delete.mockResolvedValue({ data: true });

        const response = await deletePost(123);

        expect(axios.delete).toHaveBeenCalledWith('http://localhost:8080/post/delete/123');
        expect(response.data).toBe(true);
    });

    test('updatePost should send PUT request with JSON', async () => {
        const post = { postid: 1, post_title: 'Updated Title' };
        axios.put.mockResolvedValue({ data: true });

        const response = await updatePost(post);

        expect(axios.put).toHaveBeenCalledWith(
            'http://localhost:8080/post/update',
            JSON.stringify(post),
            { headers: { 'Content-Type': 'application/json' } }
        );
        expect(response.data).toBe(true);
    });

    test('searchPosts should send GET request with keyword', async () => {
        const mockData = { data: [{ post_title: 'Keyword Match' }] };
        axios.get.mockResolvedValue(mockData);

        const response = await searchPosts('test');

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/post/search?keyword=test');
        expect(response.data).toEqual(mockData.data);
    });

    test('likePost should send PUT request', async () => {
        axios.put.mockResolvedValue({ data: true });

        const response = await likePost(1);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/post/like/1');
        expect(response.data).toBe(true);
    });

    test('unlikePost should send PUT request', async () => {
        axios.put.mockResolvedValue({ data: true });

        const response = await unlikePost(1);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/post/unlike/1');
        expect(response.data).toBe(true);
    });

    test('pinPost should send PUT request', async () => {
        axios.put.mockResolvedValue({ data: true });

        const response = await pinPost(1);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/post/pin/1');
        expect(response.data).toBe(true);
    });

    test('unpinPost should send PUT request', async () => {
        axios.put.mockResolvedValue({ data: true });

        const response = await unpinPost(1);

        expect(axios.put).toHaveBeenCalledWith('http://localhost:8080/post/unpin/1');
        expect(response.data).toBe(true);
    });

    test('findPostsByUserId should fetch user posts', async () => {
        const mockData = { data: [{ postid: 1, userid: 2 }] };
        axios.get.mockResolvedValue(mockData);

        const response = await findPostsByUserId(2);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/post/findByUserid?userid=2');
        expect(response.data).toEqual(mockData.data);
    });

    test('findPinnedPosts should fetch pinned posts', async () => {
        const mockData = { data: [{ postid: 1, is_pinned: 1 }] };
        axios.get.mockResolvedValue(mockData);

        const response = await findPinnedPosts();

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/post/findPinned');
        expect(response.data).toEqual(mockData.data);
    });

    test('getAllPostsSorted should fetch all posts sorted', async () => {
        const mockData = { data: [{ postid: 1 }] };
        axios.get.mockResolvedValue(mockData);

        const response = await getAllPostsSorted();

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/post/all');
        expect(response.data).toEqual(mockData.data);
    });

    test('getPostById should fetch post by ID', async () => {
        const mockData = { data: { postid: 1 } };
        axios.get.mockResolvedValue(mockData);

        const response = await getPostById(1);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/post/get/1');
        expect(response.data).toEqual(mockData.data);
    });
});
