// src/api/__tests__/request.test.js
const axios = require('axios');
const {
    createRequest,
    updateMoney,
    updateLoaduserByName,
    updateTorrentid,
    getTorrentid,
    deleteRequest,
    findByName,
    findByUserid,
    findByLoaduser,
    getTotalMoneyByName,
    getAllRequests,
    getInfoByRequestId,
} = require('../request');

jest.mock('axios');

describe('Request API Tests', () => {
    beforeEach(() => {
        jest.clearAllMocks();
    });

    test('createRequest should post form data with correct headers', async () => {
        const formData = new FormData();
        formData.append('title', 'Help');
        axios.post.mockResolvedValue({ data: { success: true } });

        const res = await createRequest(formData);

        expect(axios.post).toHaveBeenCalledWith(
            'http://localhost:8080/request/create',
            formData,
            { headers: { 'Content-Type': 'multipart/form-data' } }
        );
        expect(res.data).toEqual({ success: true });
    });

    test('updateMoney should send PUT request with money param', async () => {
        axios.put.mockResolvedValue({ data: true });

        const res = await updateMoney(1, 100);

        expect(axios.put).toHaveBeenCalledWith(
            'http://localhost:8080/request/updateMoney/1',
            null,
            { params: { money: 100 } }
        );
        expect(res.data).toBe(true);
    });

    test('updateLoaduserByName should send POST request with params', async () => {
        axios.post.mockResolvedValue({ data: true });

        const res = await updateLoaduserByName('testname', 10);

        expect(axios.post).toHaveBeenCalledWith(
            'http://localhost:8080/request/updateLoaduserByName',
            null,
            { params: { name: 'testname', loaduser: 10 } }
        );
        expect(res.data).toBe(true);
    });

    test('updateTorrentid should send PUT request with params', async () => {
        axios.put.mockResolvedValue({ data: true });

        const res = await updateTorrentid(5, 'torrent123');

        expect(axios.put).toHaveBeenCalledWith(
            'http://localhost:8080/request/updateTorrentid/5',
            null,
            { params: { torrentid: 'torrent123' } }
        );
        expect(res.data).toBe(true);
    });

    test('getTorrentid should fetch torrentid successfully', async () => {
        axios.get.mockResolvedValue({ data: 'torrent123' });

        const res = await getTorrentid(5);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/getTorrentid/5');
        expect(res).toBe('torrent123');
    });

    test('getTorrentid should return null on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await getTorrentid(5);

        expect(res).toBeNull();
    });

    test('deleteRequest should send DELETE request', async () => {
        axios.delete.mockResolvedValue({ data: true });

        const res = await deleteRequest(3);

        expect(axios.delete).toHaveBeenCalledWith('http://localhost:8080/request/delete/3');
        expect(res.data).toBe(true);
    });

    test('findByName should return array on success', async () => {
        axios.get.mockResolvedValue({ data: [{ id: 1 }] });

        const res = await findByName('Alice');

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/findByName', {
            params: { name: 'Alice' },
        });
        expect(res).toEqual([{ id: 1 }]);
    });

    test('findByName should return empty array on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await findByName('Alice');

        expect(res).toEqual([]);
    });

    test('findByUserid should return array on success', async () => {
        axios.get.mockResolvedValue({ data: [{ id: 2 }] });

        const res = await findByUserid(10);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/findByUserid', {
            params: { userid: 10 },
        });
        expect(res).toEqual([{ id: 2 }]);
    });

    test('findByUserid should return empty array on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await findByUserid(10);

        expect(res).toEqual([]);
    });

    test('findByLoaduser should return array on success', async () => {
        axios.get.mockResolvedValue({ data: [{ id: 3 }] });

        const res = await findByLoaduser(20);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/findByLoaduser', {
            params: { loaduser: 20 },
        });
        expect(res).toEqual([{ id: 3 }]);
    });

    test('findByLoaduser should return empty array on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await findByLoaduser(20);

        expect(res).toEqual([]);
    });

    test('getTotalMoneyByName should return number on success', async () => {
        axios.get.mockResolvedValue({ data: 123 });

        const res = await getTotalMoneyByName('Bob');

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/totalMoneyByName', {
            params: { name: 'Bob' },
        });
        expect(res).toBe(123);
    });

    test('getTotalMoneyByName should return 0 on error or invalid data', async () => {
        axios.get.mockResolvedValue({ data: 'NaN' });

        const res1 = await getTotalMoneyByName('Bob');
        expect(res1).toBe(0);

        axios.get.mockRejectedValue(new Error('fail'));
        const res2 = await getTotalMoneyByName('Bob');
        expect(res2).toBe(0);
    });

    test('getAllRequests should return array on success', async () => {
        axios.get.mockResolvedValue({ data: [{ id: 4 }] });

        const res = await getAllRequests();

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/all');
        expect(res).toEqual([{ id: 4 }]);
    });

    test('getAllRequests should return empty array on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await getAllRequests();

        expect(res).toEqual([]);
    });

    test('getInfoByRequestId should return object on success', async () => {
        axios.get.mockResolvedValue({ data: { torrentid: 'abc', money: 100, loaduser: 10 } });

        const res = await getInfoByRequestId(8);

        expect(axios.get).toHaveBeenCalledWith('http://localhost:8080/request/info/8');
        expect(res).toEqual({ torrentid: 'abc', money: 100, loaduser: 10 });
    });

    test('getInfoByRequestId should return empty object on error', async () => {
        axios.get.mockRejectedValue(new Error('fail'));

        const res = await getInfoByRequestId(8);

        expect(res).toEqual({});
    });
});
