import React, { useEffect, useState } from 'react';
import {
    Table,
    Button,
    Modal,
    message,
    Tag,
    Space,
    Tooltip,
} from 'antd';
import { ExclamationCircleOutlined } from '@ant-design/icons';
import {
    getAllComplains,
    deleteComplain,
    // 预留：你后续可以新增处理投诉的API
} from '../api/complain';
import { useNavigate } from 'react-router-dom';

const { confirm } = Modal;

const ComplainAdminPanel = () => {
    const [complains, setComplains] = useState([]);
    const [loading, setLoading] = useState(false);
    const navigate = useNavigate();

    const fetchComplains = async () => {
        setLoading(true);
        try {
            const data = await getAllComplains();
            setComplains(data);
        } catch (error) {
            message.error('获取投诉记录失败');
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchComplains();
    }, []);

    const showDeleteConfirm = (complainid) => {
        confirm({
            title: '确认删除该投诉记录吗？',
            icon: <ExclamationCircleOutlined />,
            okText: '删除',
            okType: 'danger',
            cancelText: '取消',
            onOk() {
                handleDelete(complainid);
            },
        });
    };

    const handleDelete = async (complainid) => {
        try {
            const success = await deleteComplain(complainid);
            if (success) {
                message.success('删除成功');
                fetchComplains();
            } else {
                message.error('删除失败');
            }
        } catch {
            message.error('删除请求失败');
        }
    };

    const handleProcess = (complain) => {
        const { complainid, duser, torrentid } = complain;
        navigate(`/complain-process/${complainid}`, {
            state: { complainid, duser, torrentid },
        });
    };

    const columns = [
        {
            title: '投诉ID',
            dataIndex: 'complainid',
            key: 'complainid',
            width: 80,
            fixed: 'left',
        },
        {
            title: '投诉人ID',
            dataIndex: 'puse',
            key: 'puse',
            width: 120,
        },
        {
            title: '被投诉人ID',
            dataIndex: 'duser',
            key: 'duser',
            width: 120,
        },
        {
            title: '投诉内容',
            dataIndex: 'content',
            key: 'content',
            ellipsis: { showTitle: false },
            render: (text) => (
                <Tooltip placement="topLeft" title={text}>
                    {text}
                </Tooltip>
            ),
        },
        {
            title: '相关种子ID',
            dataIndex: 'torrentid',
            key: 'torrentid',
            width: 120,
            render: (val) => val ?? <Tag color="default">无</Tag>,
        },
        {
            title: '操作',
            key: 'action',
            fixed: 'right',
            width: 150,
            render: (_, record) => (
                <Space size="middle">
                    <Button type="primary" onClick={() => handleProcess(record)}>
                        处理
                    </Button>
                    <Button danger onClick={() => showDeleteConfirm(record.complainid)}>
                        删除
                    </Button>
                </Space>
            ),
        },
    ];

    return (
        <div style={{ padding: 20 }}>
            <h2 style={{ marginBottom: 20 }}>投诉管理面板</h2>
            <Table
                rowKey="complainid"
                columns={columns}
                dataSource={complains}
                loading={loading}
                scroll={{ x: 1000 }}
                pagination={{ pageSize: 10 }}
                bordered
                size="middle"
            />
        </div>
    );
};

export default ComplainAdminPanel;
